#include "iscrypta_log_fields.h"
#include "iscrypta_log_parser.h"

#include <crypta/lib/native/unixtime_parser/unixtime_parser.h>

#include <util/string/split.h>

using namespace NCrypta;

namespace {
    TStringBuf GetFieldValue(const TStringBuf& body, const TString& field) {
        const TString fieldBeginStr = field + ": \"";
        const auto& fieldBegin = body.find(fieldBeginStr);

        if (fieldBegin == TString::npos) {
            return "";
        }

        const auto& valueBegin = fieldBegin + fieldBeginStr.length();
        const auto& valueEnd = body.find('"', valueBegin);

        if (valueEnd == TString::npos) {
            return "";
        }

        return body.substr(valueBegin, valueEnd - valueBegin);
    }

    TMaybe<bool> GetOptionalFieldBoolValue(const TStringBuf& body, const TString& field) {
        const TString fieldBeginStr = field + ": ";
        const auto& fieldBegin = body.find(fieldBeginStr);

        if (fieldBegin == TString::npos) {
            return Nothing();
        }

        const auto& valueBegin = fieldBegin + fieldBeginStr.length();
        const auto& valueEnd = body.find(' ', valueBegin);

        Y_ENSURE(valueEnd != TString::npos, "Can't find end of '" << field << "' value");

        const auto& value = body.substr(valueBegin, valueEnd - valueBegin);

        if (value == "true") {
            return true;
        } else if (value == "false") {
            return false;
        } else {
            ythrow yexception() << "Invalid boolean value '" << value << "' for field '" << field <<"'";
        }
    }

    bool SplitExtIdInfo(const TStringBuf& extIdInfo, TStringBuf& tag, TStringBuf& extId) {
        return StringSplitter(extIdInfo).Split('.').Limit(2).TryCollectInto(&tag, &extId);
    }
}

bool NIscryptaLogParser::HasExtIdUploadRequest(const NYT::TNode& row) {
    using namespace NIscryptaLogFields;

    const auto& it = row.AsMap().find(REQUEST);

    if (it == row.AsMap().end() || it->second.IsNull()) {
        return false;
    }

    const auto& request = it->second.AsString();
    return request.Contains(TYPE_UPLOAD) && request.Contains(TYPE_EXT_ID);
}

NIscryptaLogParser::TExtIdMatch NIscryptaLogParser::ParseExtIdMatch(const TString& request) {
    static const TString EXT_ID_MSG_BEGIN = "ext_id {";

    const TStringBuf requestView(request.data(), request.size());
    const auto& headerBegin = requestView.find_first_of('{');
    const auto& headerEnd = requestView.find_first_of('}');

    Y_ENSURE(headerBegin != TString::npos && headerEnd != TString::npos && headerBegin < headerEnd, "Can't find header boundaries");

    const auto& header = requestView.substr(headerBegin + 1, headerEnd - headerBegin - 1);

    Y_ENSURE(header.Contains(TYPE_UPLOAD), "TYPE_UPLOAD is missing in header");

    const auto& bodyBegin = requestView.find('{', headerEnd);
    const auto& bodyEnd = requestView.find_last_of('}');

    Y_ENSURE(bodyBegin != TString::npos && bodyEnd != TString::npos && bodyBegin < bodyEnd, "Can't find body boundaries");

    const auto& body = requestView.substr(bodyBegin + 1, bodyEnd - bodyBegin - 1);

    Y_ENSURE(body.Contains(TYPE_EXT_ID), "TYPE_EXT_ID is missing in body");

    const auto& msgFieldBegin = body.find(EXT_ID_MSG_BEGIN);

    Y_ENSURE(msgFieldBegin != TString::npos, "Can't find beginning of message with external id");

    const auto& msgBegin = msgFieldBegin + EXT_ID_MSG_BEGIN.length();
    const auto& msgEnd = body.find("}", msgBegin);

    Y_ENSURE(msgEnd != TString::npos, "Can't find end of message with external id");

    const auto& extIdBody = body.substr(msgBegin, msgEnd - msgBegin - 1);
    const auto& extIdInfo = GetFieldValue(extIdBody, "ext_id");

    Y_ENSURE(!extIdInfo.empty(), "Can't find value of ext_id field or value is empty");

    const auto& yuid = GetFieldValue(extIdBody, "yuid");

    Y_ENSURE(!yuid.empty(), "Can't find value of yuid field or value is empty");

    TStringBuf tag;
    TStringBuf extId;

    SplitExtIdInfo(extIdInfo, tag, extId);

    const auto& syntheticMatch = GetOptionalFieldBoolValue(extIdBody, "synthetic_match");

    return {
        .ExtId = {
            .Tag = TString(tag),
            .Value = TString(extId)
        },
        .Yandexuid = TString(yuid),
        .Synthetic = syntheticMatch.GetOrElse(false)
    };
}

NIscryptaLogParser::TUploadInfo NIscryptaLogParser::ParseUploadInfo(const NYT::TNode& row) {
    using namespace NIscryptaLogFields;

    const auto& unixtime = NUnixtimeParser::Parse(row.At(UNIXTIME).AsString());

    Y_ENSURE(unixtime.Defined(), "Unixtime is undefined");

    const auto& extIdMatch = ParseExtIdMatch(row.At(REQUEST).AsString());

    return {.Match = std::move(extIdMatch), .Timestamp = unixtime.GetRef()};
}

bool NIscryptaLogParser::HasExtIdIdentifyRequest(const NYT::TNode& row) {
    using namespace NIscryptaLogFields;

    const auto& it = row.AsMap().find(REQUEST);

    if (it == row.AsMap().end() || it->second.IsNull()) {
        return false;
    }

    const auto& request = it->second.AsString();
    return request.Contains(TYPE_IDENTIFY) && request.Contains(EXT_ID);
}

NIscryptaLogParser::TExtId NIscryptaLogParser::ParseExtId(const TString& identifyRequest) {
    const TStringBuf requestView(identifyRequest.data(), identifyRequest.size());
    const auto& headerBegin = requestView.find_first_of('{');
    const auto& headerEnd = requestView.find_first_of('}');

    Y_ENSURE(headerBegin != TString::npos && headerEnd != TString::npos && headerBegin < headerEnd, "Can't find header boundaries");

    const auto& header = requestView.substr(headerBegin + 1, headerEnd - headerBegin - 1);

    Y_ENSURE(header.Contains(TYPE_IDENTIFY), "TYPE_IDENTIFY is missing in header");

    const auto& bodyBegin = requestView.find('{', headerEnd);
    const auto& bodyEnd = requestView.find_last_of('}');

    Y_ENSURE(bodyBegin != TString::npos && bodyEnd != TString::npos && bodyBegin < bodyEnd, "Can't find body boundaries");

    const auto& body = requestView.substr(bodyBegin + 1, bodyEnd - bodyBegin - 1);

    const auto& extIdInfo = GetFieldValue(body, "ext_id");

    Y_ENSURE(!extIdInfo.empty(), "Can't find value of ext_id field or value is empty");

    TStringBuf tag;
    TStringBuf extId;

    Y_ENSURE(SplitExtIdInfo(extIdInfo, tag, extId) && !extId.empty() && !tag.empty(), "Invalid external id");

    return {
        .Tag = TString(tag),
        .Value = TString(extId)
    };
}

bool NIscryptaLogParser::UploadInfoIsValid(const NIscryptaLogParser::TUploadInfo& info) {
    return !info.Match.ExtId.Tag.empty() && !info.Match.ExtId.Value.empty() && !info.Match.Yandexuid.empty();
}

NIscryptaLogParser::TIdentifyInfo NIscryptaLogParser::ParseIdentifyInfo(const NYT::TNode& row) {
    using namespace NIscryptaLogFields;

    const auto& unixtime = NUnixtimeParser::Parse(row.At(UNIXTIME).AsString());

    Y_ENSURE(unixtime.Defined(), "Unixtime is undefined");

    const auto& extId = ParseExtId(row.At(REQUEST).AsString());

    return {.ExtId = std::move(extId), .Timestamp = unixtime.GetRef()};
}

bool NIscryptaLogParser::TExtId::operator==(const NIscryptaLogParser::TExtId& other) const {
    return (Tag == other.Tag) && (Value == other.Value);
}

bool NIscryptaLogParser::TExtId::operator!=(const NIscryptaLogParser::TExtId& other) const {
    return !(*this == other);
}

bool NIscryptaLogParser::TExtIdMatch::operator==(const NIscryptaLogParser::TExtIdMatch& other) const {
    return (ExtId == other.ExtId) && (Yandexuid == other.Yandexuid) && (Synthetic == other.Synthetic);
}

bool NIscryptaLogParser::TExtIdMatch::operator!=(const NIscryptaLogParser::TExtIdMatch& other) const {
    return !(*this == other);
}

bool NIscryptaLogParser::TUploadInfo::operator==(const NIscryptaLogParser::TUploadInfo& other) const {
    return (Match == other.Match) && (Timestamp == other.Timestamp);
}

bool NIscryptaLogParser::TUploadInfo::operator!=(const NIscryptaLogParser::TUploadInfo& other) const {
    return !(*this == other);
}

bool NIscryptaLogParser::TIdentifyInfo::operator==(const NIscryptaLogParser::TIdentifyInfo& other) const {
    return (ExtId == other.ExtId) && (Timestamp == other.Timestamp);
}

bool NIscryptaLogParser::TIdentifyInfo::operator!=(const NIscryptaLogParser::TIdentifyInfo& other) const {
    return !(*this == other);
}
