#include "juggler_client.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/neh/http_common.h>

#include <util/string/cast.h>
#include <util/string/join.h>

using namespace NCrypta;

namespace {
    NNeh::TResponseRef RequestWithTimeout(const TString& url, TDuration timeout, const TString& content) {
        auto message = NNeh::TMessage::FromString(url);
        Y_ENSURE(NNeh::NHttp::MakeFullRequest(message, "", content, "application/json"), "Failed to build message");
        NNeh::TResponseRef response = NNeh::Request(message)->Wait(timeout);

        Y_ENSURE(response, "Timeout fetching " << url);
        Y_ENSURE(!response->IsError(), response->GetErrorText());

        return response;
    }

    const TString& OK = "OK";
    const TString& WARN = "WARN";
    const TString& CRIT = "CRIT";
}

TJugglerClient::TJugglerClient(const TJugglerClientOptions& options)
    : Options(options) {
}

void TJugglerClient::ReportOk(const TString& host, const TString& service, const TString& description) {
    ReportStatus(OK, host, service, description);
}

void TJugglerClient::ReportWarn(const TString& host, const TString& service, const TString& description) {
    ReportStatus(WARN, host, service, description);
}

void TJugglerClient::ReportCrit(const TString& host, const TString& service, const TString& description) {
    ReportStatus(CRIT, host, service, description);
}

void TJugglerClient::ReportStatus(const TString& status, const TString& host, const TString& service, const TString& description) {
    auto url = Join("/", Options.GetEndpointOptions().GetUrlPrefix(), "events");

    TStringStream content;
    NJson::TJsonWriter jsonWriter(&content, false);
    jsonWriter.OpenMap();
    jsonWriter.Write("source", Options.GetSource());

    jsonWriter.OpenArray("events");

    jsonWriter.OpenMap();
    jsonWriter.Write("description", description);
    jsonWriter.Write("host", host);
    jsonWriter.Write("service", service);
    jsonWriter.Write("status", status);
    jsonWriter.CloseMap();

    jsonWriter.CloseArray();

    jsonWriter.CloseMap();
    jsonWriter.Flush();

    auto response = RequestWithTimeout(url, TDuration::Seconds(Options.GetEndpointOptions().GetTimeoutSec()), content.Str());

    NJson::TJsonValue jsonResponse;
    NJson::ReadJsonTree(response->Data, &jsonResponse, true);
    Y_ENSURE(jsonResponse["accepted_events"].GetIntegerSafe() == 1, "Juggler did not accept event");
}
