#include "local_kv.h"

#include <util/generic/variant.h>

using namespace NCrypta;

bool TLocalKv::TryProcessReply() {
    if (ReplyQueue.empty()) {
        return false;
    }

    auto reply = ReplyQueue.front();

    if (std::holds_alternative<TGetReply>(reply)) {
        if (GetHandler) {
            auto getReply = std::get<TGetReply>(reply);
            GetHandler(getReply.RequestId, std::move(getReply.Record));
        }
    } else if (std::holds_alternative<TStoreReply>(reply)) {
        if (StoreHandler) {
            auto storeReply = std::get<TStoreReply>(reply);
            StoreHandler(storeReply.RequestId, std::move(storeReply.Key));
        }
    } else if (std::holds_alternative<TRemoveReply>(reply)) {
        if (RemoveHandler) {
            auto removeReply = std::get<TRemoveReply>(reply);
            RemoveHandler(removeReply.RequestId, std::move(removeReply.Key));
        }
    } else if (std::holds_alternative<TErrorReply>(reply)) {
        if (ErrorHandler) {
            auto& result = std::get<TErrorReply>(reply);
            ErrorHandler(result.RequestId, std::move(result.Key), std::move(result.Error));
        }
    }

    ReplyQueue.pop_front();

    return true;
}

IDatabase::TRequestId TLocalKv::Get(const TString& key, const TDuration& ttl) {
    Y_UNUSED(ttl);

    auto requestId = NextRequestId++;

    if (Storage.contains(key)) {
        ReplyQueue.emplace_back(TGetReply{requestId, TRecord{.Key = key, .Value = Storage.at(key)}});
    } else {
        ReplyQueue.emplace_back(TGetReply{requestId, Nothing()});
    }

    return requestId;
}

IDatabase::TRequestId TLocalKv::Store(const TRecord& record, const TDuration& ttl) {
    Y_UNUSED(ttl);

    auto requestId = NextRequestId++;
    Storage[record.Key] = record.Value;
    ReplyQueue.emplace_back(TStoreReply{requestId, record.Key});

    return requestId;
}

IDatabase::TRequestId TLocalKv::Remove(const TString& key, ui64 cas) {
    Y_UNUSED(cas);

    auto requestId = NextRequestId++;

    if (Storage.contains(key)) {
        Storage.erase(key);
        ReplyQueue.emplace_back(TRemoveReply{requestId, key});
    } else {
        ReplyQueue.emplace_back(TErrorReply{requestId, key, TString("Storage has no key ") + key});
    }

    return requestId;
}

void TLocalKv::SetGetHandler(const TGetHandler& handler) {
    GetHandler = handler;
}

void TLocalKv::SetStoreHandler(const TStoreHandler& handler) {
    StoreHandler = handler;
}

void TLocalKv::SetRemoveHandler(const TRemoveHandler& handler) {
    RemoveHandler = handler;
}

void TLocalKv::SetErrorHandler(const TErrorHandler& handler) {
    ErrorHandler = handler;
}
