#include <crypta/lib/native/local_kv/local_kv.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/maybe.h>
#include <util/generic/xrange.h>

#include <functional>

Y_UNIT_TEST_SUITE(LocalKv) {
    using TRequestId = NCrypta::IDatabase::TRequestId;
    using TOutcomes = THashMap<TRequestId, TString>;

    void AddCallbacks(NCrypta::TLocalKv& kv, TOutcomes& outcomes) {
        auto onGet = [&outcomes](TRequestId id, TMaybe<NCrypta::TRecord>&& record) {
            outcomes.emplace(id,
                    record.Defined() ? (TString("Found ") + record->Key + " = " + record->Value) : TString("Not Found"));
        };

        auto onStore = [&outcomes](TRequestId id, TString&& key) {
            outcomes.emplace(id, "Stored " + key);
        };

        auto onRemove = [&outcomes](TRequestId id, TString&& key) {
            outcomes.emplace(id, "Removed " + key);
        };

        auto onError = [&outcomes](TRequestId id, TString&& key, TString&& error) {
            outcomes.emplace(id, key + " : " + error);
        };

        kv.SetGetHandler(onGet);
        kv.SetStoreHandler(onStore);
        kv.SetRemoveHandler(onRemove);
        kv.SetErrorHandler(onError);
    }

    void PrepareKv(NCrypta::TLocalKv& kv, TOutcomes& outcomes) {
        UNIT_ASSERT(!kv.TryProcessReply());
        AddCallbacks(kv, outcomes);
    }

    void CheckKvAndOutcomes(NCrypta::TLocalKv& kv, const TOutcomes& outcomes, const TOutcomes& refs) {
        for (auto i : xrange(refs.size())) {
            UNIT_ASSERT_C(kv.TryProcessReply(), ToString(i));
        }
        UNIT_ASSERT(!kv.TryProcessReply());
        UNIT_ASSERT_EQUAL(refs, outcomes);

    }

    Y_UNIT_TEST(Empty) {
        NCrypta::TLocalKv kv;
        TOutcomes outcomes;
        PrepareKv(kv, outcomes);

        TOutcomes refs;
        refs[kv.Get("foo")] = "Not Found";
        refs[kv.Remove("bar")] = "bar : Storage has no key bar";

        CheckKvAndOutcomes(kv, outcomes, refs);
    }

    Y_UNIT_TEST(SetGetRemove) {
        NCrypta::TLocalKv kv;
        TOutcomes outcomes;
        PrepareKv(kv, outcomes);

        TOutcomes refs;
        refs[kv.Store(NCrypta::TRecord{.Key = "foo", .Value = "bar"})] = "Stored foo";
        refs[kv.Get("foo")] = "Found foo = bar";
        refs[kv.Remove("foo")] = "Removed foo";
        refs[kv.Get("foo")] = "Not Found";
        refs[kv.Remove("foo")] = "foo : Storage has no key foo";

        CheckKvAndOutcomes(kv, outcomes, refs);
    }
}
