#include "log.h"

#include <crypta/lib/native/log/proto/sink_config.pb.h>

#include <library/cpp/logger/priority.h>

#include <spdlog/async.h>
#include <spdlog/sinks/rotating_file_sink.h>
#include <spdlog/sinks/stdout_sinks.h>

#include <library/cpp/json/json_writer.h>

#include <util/datetime/base.h>
#include <util/generic/algorithm.h>
#include <util/stream/str.h>
#include <util/system/thread.h>


using namespace NCrypta;
using namespace NCrypta::NLog;

namespace {
    const std::string DEFAULT_LOG_NAME = "__default_logger";
    TLogPtr DEFAULT_LOG = spdlog::stderr_logger_mt(DEFAULT_LOG_NAME);

    TString GetUpperCaseLogLevel(spdlog::level::level_enum level) {
        auto levelView = spdlog::level::to_string_view(level);
        TString result;
        result.reserve(levelView.size());
        Transform(levelView.begin(), levelView.end(), std::back_inserter(result), [](char c){ return toupper(c); });
        return result;
    }

    class TDeployFormatter : public spdlog::formatter {
    public:
        void format(const spdlog::details::log_msg& msg, spdlog::memory_buf_t& dest) override {
            auto logTime = TInstant::MicroSeconds(std::chrono::time_point_cast<std::chrono::microseconds>(msg.time).time_since_epoch().count());

            TStringStream output;
            NJson::TJsonWriter jsonWriter(&output, false);
            jsonWriter.OpenMap();
            jsonWriter.Write("message", TStringBuf(msg.payload.data(), msg.payload.size()));
            jsonWriter.Write("levelStr", GetUpperCaseLogLevel(msg.level));
            jsonWriter.Write("loggerName", TStringBuf(msg.logger_name.data(), msg.logger_name.size()));
            jsonWriter.Write("@timestamp", logTime.ToString());
            jsonWriter.CloseMap();
            jsonWriter.Flush();

            dest.append(output.Str());
            dest.push_back('\n');
        };

        virtual std::unique_ptr<formatter> clone() const override {
            return spdlog::details::make_unique<TDeployFormatter>();
        };
    };

    std::shared_ptr<spdlog::details::thread_pool> GetThreadPool() {
        if (spdlog::thread_pool() == nullptr) {
            spdlog::init_thread_pool(1 << 16, 1);
            spdlog::flush_every(std::chrono::seconds(1));
        }

        return spdlog::thread_pool();
    }

    spdlog::sink_ptr CreateSink(const TSinkConfig& config) {
        spdlog::sink_ptr result;

        if (config.GetType() == "stdout") {
            result = std::make_shared<spdlog::sinks::stdout_sink_mt>();
        }
        if (config.GetType() == "stderr") {
            result = std::make_shared<spdlog::sinks::stderr_sink_mt>();
        }
        if (config.GetType() == "rotating") {
            Y_ENSURE(config.HasFile(), "Rotating sink must have file specified");
            Y_ENSURE(config.HasMaxFileSize(), "Rotating sink must have file_size specified");
            Y_ENSURE(config.HasMaxFiles(), "Rotating sink must have max_files specified");

            result = std::make_shared<spdlog::sinks::rotating_file_sink_mt>(config.GetFile(), config.GetMaxFileSize(), config.GetMaxFiles());
        }

        if (!result) {
            throw spdlog::spdlog_ex("Unknown log type: " + config.GetType());
        }

        if (config.HasFormat()) {
            result->set_pattern(config.GetFormat());
        } else if (config.GetDeployFormat()) {
            result->set_formatter(std::make_unique<TDeployFormatter>());
        }
        return result;
    }

    TLogPtr CreateLogger(const std::string& name, const TLogConfig& config) {
        Y_ENSURE(!config.GetTargets().empty(), "Logger " << name << " has no targets");

        std::vector<spdlog::sink_ptr> sinks;

        for (const auto& target : config.GetTargets()) {
            if (target.GetType() == "devnull") {
                continue;
            }

            sinks.push_back(CreateSink(target));
        }

        auto logger = std::make_shared<spdlog::async_logger>(name, sinks.begin(), sinks.end(), GetThreadPool(), spdlog::async_overflow_policy::block);
        spdlog::register_logger(logger);

        return logger;
    }

    void SetCommonParams(TLogPtr log, const TLogConfig& config) {
        if (config.HasFormat()) {
            log->set_pattern(config.GetFormat());
        } else if (config.GetDeployFormat()) {
            log->set_formatter(std::make_unique<TDeployFormatter>());
        }
        if (config.HasLevel()) {
            log->set_level(spdlog::level::from_str(config.GetLevel()));
        }
    }
}

namespace NCrypta::NLog {
    void RegisterLog(const std::string& logName, const TLogConfig& config) {
        GetLog()->info("Found log: {}", logName);

        auto logger = CreateLogger(logName, config);
        SetCommonParams(logger, config);
    }

    void RegisterLogs(const TLogConfigs& config) {
        for (const auto& [logName, logConfig] : config) {
            RegisterLog(logName, logConfig);
        }
    }

    TLogPtr GetLog(const TString& name) {
        auto res = spdlog::get(name);
        if (res == nullptr) {
            ythrow yexception() << "Log not registered: " << name;
        }
        return res;
    }

    TLogPtr GetLog() {
        return DEFAULT_LOG;
    }

    spdlog::level::level_enum ConvertArcadiaToSpdlogLevel(int level) {
        switch (level) {
            case TLOG_RESOURCES:
                return spdlog::level::trace;
            case TLOG_DEBUG:
                return spdlog::level::debug;
            case TLOG_INFO:
            case TLOG_NOTICE:
                return spdlog::level::info;
            case TLOG_WARNING:
                return spdlog::level::warn;
            case TLOG_ERR:
                return spdlog::level::err;
            case TLOG_ALERT:
            case TLOG_CRIT:
            case TLOG_EMERG:
                return spdlog::level::critical;
            default:
                ythrow yexception() << "unknown verbose level: " << level;
        }
    }
}
