#include "logbroker.h"

#include <util/string/cast.h>
#include <util/string/split.h>

using NPersQueue::ICredentialsProvider;
using NPersQueue::ILogger;
using NPersQueue::TConsumerSettings;
using NPersQueue::TPQLibSettings;

using namespace NCryptaLogbroker;
using namespace NCryptaLogbroker::NLogbrokerMetrics;

std::shared_ptr<ICredentialsProvider> NCryptaLogbroker::CreateTvmProvider(
    ui32 serverTvmId,
    ui32 clientTvmId,
    const TString& clientTvmSecret,
    const TString& alias,
    TIntrusivePtr<NPersQueue::ILogger>& logger,
    int logPriority) {
    NTvmAuth::NTvmApi::TClientSettings tvmSettings;

    tvmSettings.SetSelfTvmId(clientTvmId);
    tvmSettings.EnableServiceTicketsFetchOptions(
        clientTvmSecret,
        {{alias, serverTvmId}});

    auto tvmLogger = MakeIntrusive<NTvmAuth::TCerrLogger>(logPriority);
    auto tvmClient = std::make_shared<NTvmAuth::TTvmClient>(tvmSettings, tvmLogger);
    return CreateTVMCredentialsProvider(tvmClient, logger, alias);
}

void TLogbrokerPuller::ProcessMessage(const TMessage& msg, const TOnMessage& callBack) {
    switch (msg.GetValue().Type) {
        case NPersQueue::EMT_DATA: {
            const auto& responseData = msg.GetValue().Response.GetData();

            for (ui32 i = 0; i < responseData.MessageBatchSize(); ++i) {
                const auto& t = responseData.GetMessageBatch(i);

                for (ui32 j = 0; j < t.MessageSize(); ++j) {
                    const auto& m = t.GetMessage(j);
                    const auto& meta = m.GetMeta();

                    if (Splitter) {
                        for (const auto& it : StringSplitter(m.GetData()).Split(*Splitter).SkipEmpty()) {
                            callBack(ToString(it.Token()));
                            Metrics.CounterIncrement(TOTAL_STRINGS);
                        }
                    } else {
                        callBack(m.GetData());
                        Metrics.CounterIncrement(TOTAL_STRINGS);
                    }

                    Metrics.HistogramRecord(LB_LAG, TInstant::Now().MilliSeconds() - meta.GetCreateTimeMs());
                    Metrics.CounterIncrement(TOTAL_MESSAGES);
                }
                Metrics.RateIncrement(DATA_RPS, t.MessageSize());
            }

            Consumer->Commit({responseData.GetCookie()});
            break;
        }

        case NPersQueue::EMT_ERROR: {
            TGuard<TMutex> Lock(ConsumerInUseLock);

            if (Consumer->IsDead().HasValue()) {
                Consumer = CreateConsumer();
                Metrics.CounterIncrement(CONSUMER_ERRORS);
            }
            break;
        }

        case NPersQueue::EMT_LOCK: {
            auto promise = msg.GetValue().ReadyToRead;
            promise.SetValue(NPersQueue::TLockInfo{0, 0, false});
            Metrics.SetIGauge(PARTITIONS_LOCKED, Metrics.GetIGauge(PARTITIONS_LOCKED) + 1);
            break;
        }
        case NPersQueue::EMT_RELEASE:
            Metrics.SetIGauge(PARTITIONS_LOCKED, Metrics.GetIGauge(PARTITIONS_LOCKED) - 1);
            break;
        case NPersQueue::EMT_COMMIT:
            Metrics.CounterIncrement(COMMIT_MESSAGES);
            Metrics.SetIGauge(LAST_COMMIT_TIME, TInstant::Now().Seconds());
            break;

        default:
            Y_ASSERT(0);
    }
}

void TLogbrokerPuller::Process(const TOnMessage& callBack, const TNeedStop& needStop, const TDuration timeout) {
    /* @alexnick: GetNextMessage возвращает фьючу, которую нельзя терять. Надо ждать ее столько, сколько потребуется.
    Вообще таймаутов на твоей стороне быть не должно - если с сессией что-то случается, очередной GetNextMessage засигналится ошибкой.
    */
    NThreading::TFuture<NPersQueue::TConsumerMessage> msg;

    {
        TGuard<TMutex> Lock(ConsumerInUseLock);
        msg = Consumer->GetNextMessage();
    }

    while (!needStop()) {
        if (!msg.Wait(timeout)) {
            Metrics.CounterIncrement(TIMEOUTS);
            continue;
        }

        ProcessMessage(msg, callBack);

        {
            TGuard<TMutex> Lock(ConsumerInUseLock);
            msg = Consumer->GetNextMessage();
        }
    }
}
