#pragma once

#include <kikimr/persqueue/sdk/deprecated/cpp/v2/persqueue.h>
#include <crypta/lib/native/metrics_helper/metrics_helper.h>
#include <util/system/mutex.h>

namespace NCryptaLogbroker {
    using TOnMessage = std::function<void(const TStringBuf data)>;
    using TNeedStop = std::function<bool()>;
    using TMessage = NThreading::TFuture<NPersQueue::TConsumerMessage>;

    using NCryptaSolomonMetrics::EMetricType;
    using NCryptaSolomonMetrics::TMetricsHelper;
    using NCryptaSolomonMetrics::TMetricsInit;

    namespace NLogbrokerMetrics {
        const TString PARTITIONS_LOCKED{"lb_partitions_locked"};
        const TString TOTAL_STRINGS{"lb_total_strings"};
        const TString TOTAL_MESSAGES{"lb_total_data_messages"};
        const TString DATA_RPS{"lb_data_messages_rps"};
        const TString CONSUMER_ERRORS{"lb_consumer_errors"};
        const TString COMMIT_MESSAGES{"lb_commit_messages"};
        const TString LAST_COMMIT_TIME{"lb_last_commit_time"};
        const TString TIMEOUTS{"lb_wait_timeouts"};
        const TString LB_LAG{"lb_lag"};
    }

    const TMetricsInit InitMetrics = {
        {NLogbrokerMetrics::CONSUMER_ERRORS, EMetricType::COUNTER, {}},
        {NLogbrokerMetrics::COMMIT_MESSAGES, EMetricType::COUNTER, {}},
        {NLogbrokerMetrics::TIMEOUTS, EMetricType::COUNTER, {}},
        {NLogbrokerMetrics::TOTAL_MESSAGES, EMetricType::COUNTER, {}},
        {NLogbrokerMetrics::TOTAL_STRINGS, EMetricType::COUNTER, {}},
        {NLogbrokerMetrics::PARTITIONS_LOCKED, EMetricType::IGAUGE, {}},
        {NLogbrokerMetrics::LAST_COMMIT_TIME, EMetricType::IGAUGE, {}},
        {NLogbrokerMetrics::DATA_RPS, EMetricType::RATE, {}},
        {NLogbrokerMetrics::LB_LAG, EMetricType::HISTOGRAM, {}, NMonitoring::TBucketBounds{100, 200, 400, 800, 1600, 3200, 6400}}};

    class TLogbrokerPuller {
    private:
        NPersQueue::TPQLib PQ;
        NPersQueue::TConsumerSettings ConsumerSettings;
        TIntrusivePtr<NPersQueue::ILogger> Logger = nullptr;
        TString SessionId;
        TMetricsHelper Metrics;
        THolder<NPersQueue::IConsumer> Consumer;
        TMutex ConsumerInUseLock;
        TMaybe<char> Splitter;

        void ProcessMessage(const TMessage&, const TOnMessage&);

        THolder<NPersQueue::IConsumer> CreateConsumer() {
            auto consumer = PQ.CreateConsumer(ConsumerSettings, Logger, true);
            auto startFuture = consumer->Start();

            startFuture.GetValueSync();
            Y_ENSURE(!consumer->IsDead().HasValue());

            SessionId = startFuture.GetValue().Response.GetInit().GetSessionId();

            Metrics.SetIGauge(NLogbrokerMetrics::PARTITIONS_LOCKED, 0);

            return consumer;
        }

    public:
        TLogbrokerPuller(
            const NPersQueue::TPQLibSettings& pqSettings,
            const NPersQueue::TConsumerSettings& consumerSettings,
            TIntrusivePtr<NPersQueue::ILogger>& logger,
            NMonitoring::TLabels commonLabels = {},
            TMaybe<char> splitter = '\n')
            : PQ(pqSettings)
            , ConsumerSettings(consumerSettings)
            , Logger(logger)
            , Metrics(commonLabels, InitMetrics)
            , Consumer(CreateConsumer())
            , Splitter(splitter)
            {};

        void Process(const TOnMessage&, const TNeedStop&, const TDuration timeout = TDuration::Seconds(1));
        const TString& GetSessionId() const {
            return SessionId;
        }
        const TMetricsHelper& GetMetrics() const {
            return Metrics;
        }
    };

    std::shared_ptr<NPersQueue::ICredentialsProvider> CreateTvmProvider(ui32, ui32, const TString&, const TString&, TIntrusivePtr<NPersQueue::ILogger>&, int logLevel = 3);
}
