#pragma once

#include <library/cpp/any/any.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NCryptaSolomonMetrics {
    enum class EMetricType {
        COUNTER,
        RATE,
        HISTOGRAM,
        IGAUGE,
        DGAUGE
    };

    using NMonitoring::TLabels;

    struct TMetricWrapper {
        TString Name;
        EMetricType Type;
        TLabels Labels;
        NAny::TAny Args;
        TMetricWrapper(const TString& name, const EMetricType type, const TLabels& labels)
            : Name(name)
            , Type(type)
            , Labels(labels)
        {
            Labels.Add("sensor", Name);
        }
        TMetricWrapper(const TString& name, const EMetricType type, const TLabels& labels, const NAny::TAny& args)
            : Name(name)
            , Type(type)
            , Labels(labels)
            , Args(args)
        {
            Labels.Add("sensor", Name);
        }
    };

    using TMetricsInit = TVector<TMetricWrapper>;

    class TMetricsHelper {
    private:
        THashMap<TString, NMonitoring::IMetric*> MetricsNames;
        NMonitoring::TMetricRegistry Metrics;

    public:
        TMetricsHelper(const TLabels& commonLabels, const TMetricsInit& initLabels)
        : Metrics(commonLabels)
        {
            for (auto& metric : initLabels) {
                switch (metric.Type) {
                    case EMetricType::COUNTER:
                        MetricsNames[metric.Name] = Metrics.Counter(metric.Labels);
                        break;
                    case EMetricType::RATE:
                        MetricsNames[metric.Name] = Metrics.Rate(metric.Labels);
                        break;
                    case EMetricType::HISTOGRAM:
                        MetricsNames[metric.Name] = Metrics.HistogramRate(
                            metric.Labels,
                            NMonitoring::ExplicitHistogram(metric.Args.Cast<NMonitoring::TBucketBounds>()));
                        break;
                    case EMetricType::IGAUGE:
                        MetricsNames[metric.Name] = Metrics.IntGauge(metric.Labels);
                        break;
                    case EMetricType::DGAUGE:
                        MetricsNames[metric.Name] = Metrics.Gauge(metric.Labels);
                        break;
                    default:
                        Y_ENSURE(false, "Metric not supported");
                }
            }
        }

        /* helper function to increment metrics by name */
        void CounterIncrement(const TString& sensorName, ui64 value = 1) {
            dynamic_cast<NMonitoring::TCounter*>(MetricsNames.at(sensorName))->Add(value);
        }
        void RateIncrement(const TString& sensorName, ui64 value = 1) {
            dynamic_cast<NMonitoring::TRate*>(MetricsNames.at(sensorName))->Add(value);
        }
        void HistogramRecord(const TString& sensorName, ui64 value) {
            dynamic_cast<NMonitoring::THistogram*>(MetricsNames.at(sensorName))->Record(value);
        }
        void SetIGauge(const TString& sensorName, ui64 value) {
            dynamic_cast<NMonitoring::TIntGauge*>(MetricsNames.at(sensorName))->Set(value);
        }
        i64 IncIGauge(const TString& sensorName) {
            return dynamic_cast<NMonitoring::TIntGauge*>(MetricsNames.at(sensorName))->Inc();
        }
        i64 DecIGauge(const TString& sensorName) {
            return dynamic_cast<NMonitoring::TIntGauge*>(MetricsNames.at(sensorName))->Dec();
        }
        ui64 GetIGauge(const TString& sensorName) const {
            return dynamic_cast<NMonitoring::TIntGauge*>(MetricsNames.at(sensorName))->Get();
        }

        TString Solomonify() const;
        TString Textify() const;
    };
}
