#include "buffered_producer.h"

#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/time/sleeper.h>

#include <util/system/thread.h>

using namespace NCrypta;
using namespace NCrypta::NPQ;

namespace {
    const char MESSAGE_DELIMITER = '\n';
}

TBufferedProducer::TBufferedProducer(NPersQueue::TPQLib& pqLib,
                                     const NPersQueue::TProducerSettings& producerSettings,
                                     size_t maxChunkSize,
                                     TDuration maxFlushInterval,
                                     size_t maxInflightBytes,
                                     TIntrusivePtr<NPersQueue::ILogger>& logger,
                                     NLog::TLogPtr dropLog,
                                     const TStats::TSettings& statsSettings)
    : DropLog(dropLog)
    , Stats(MakeGraphiteMetric(
                "pq.producer",
                CanonizeGraphiteMetricNode(producerSettings.Topic),
                producerSettings.PartitionGroup ? ToString(producerSettings.PartitionGroup) : CanonizeGraphiteMetricNode(producerSettings.SourceId)
            ), statsSettings)
    , Producer(pqLib, producerSettings, logger, Stats)
    , MaxChunkSize(maxChunkSize)
    , MaxFlushInterval(maxFlushInterval)
    , MaxInFlightBytes(maxInflightBytes)
{
}

void TBufferedProducer::Start(TDuration timeout) {
    Producer.Start(timeout);

    AtomicSet(IsRunning, 1);

    Thread = SystemThreadFactory()->Run([this] {
        TThread::SetCurrentThreadName("TBufferedProducer");
        Run();
    });
}

void TBufferedProducer::Run() {
    TSleeper sleeper(100, TDuration::MicroSeconds(100));

    while (AtomicGet(IsRunning)) {
        TString message;
        if (MessagesQueue.Dequeue(message)) {
            Buffer << message << MESSAGE_DELIMITER;
            sleeper.Wake();
        } else {
            sleeper.Sleep();
        }

        if ((Buffer.size() >= MaxChunkSize) || ((TInstant::Now() - LastFlushTime) > MaxFlushInterval)) {
            Flush();
        }
    }
}

bool TBufferedProducer::TryEnqueue(TString message) {
    // Y_ENSURE(AllOf(message, [](char c) { return c != MESSAGE_DELIMITER; }), "Message '" << message << "' contains delimiter"); TODO(levanov): Not compatible with spdlog pqlib_sink

    const auto& size = message.size() + sizeof(MESSAGE_DELIMITER);

    if ((InFlightBytes.Add(size) > MaxInFlightBytes) && (MaxInFlightBytes > 0)) {
        InFlightBytes.Sub(size);

        if (DropLog) {
            DropLog->info(message);
        }

        return false;
    }

    UpdateInFlightBytesStat();

    MessagesQueue.Enqueue(std::move(message));

    return true;
}

void TBufferedProducer::Flush() {
    if (Buffer.empty()) {
        return;
    }

    const auto size = Buffer.size();

    Producer.Write(Buffer).Subscribe([this, size](const auto& /* future */) {
        InFlightBytes.Sub(size);
        UpdateInFlightBytesStat();
    });

    Buffer.clear();
    LastFlushTime = TInstant::Now();
}

void TBufferedProducer::UpdateInFlightBytesStat() {
    Stats.Max->Add("inflight.bytes.max", InFlightBytes.Val());
}

void TBufferedProducer::Stop(TDuration timeout) {
    auto deadline = timeout.ToDeadLine();

    while ((InFlightBytes.Val() > 0) && (TInstant::Now() < deadline)) {
        Sleep(TDuration::MilliSeconds(100));
    }

    if (AtomicCas(&IsRunning, 0, 1)) {
        Thread->Join();
    }
}

TBufferedProducer::~TBufferedProducer() {
    Stop();
}
