#include "producer.h"

#include <crypta/lib/native/log/wrappers/pqlib/pq_spdlog_wrapper.h>
#include <crypta/lib/native/password_file/password_file.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>

#include <util/random/random.h>
#include <util/string/builder.h>

using namespace NCrypta;
using namespace NCrypta::NPQ;

namespace {
    TString GetRandomSource(const TString& prefix) {
        return TStringBuilder() << prefix << "_" << TInstant::Now().Seconds() << "_" << RandomNumber<ui64>();
    }

    NPersQueue::TProducerSettings GetProducerSettings(const TProducerConfig& config, const std::shared_ptr<NPersQueue::ICredentialsProvider>& credentialsProvider) {
        return NPersQueue::TProducerSettings {
                .Server = NPersQueue::TServerSetting(config.GetServer(), config.GetPort()),
                .CredentialsProvider = credentialsProvider,
                .Topic = config.GetTopic(),
                .ReconnectOnFailure = true,
        };
    }
}

TProducer::TProducer(
        TAtomicSharedPtr<NPersQueue::TPQLib> pqLib,
        const TProducerConfig& config,
        const std::shared_ptr<NPersQueue::ICredentialsProvider>& credentialsProvider,
        const TStats::TSettings& statsSettings)
    : PqLib(std::move(pqLib))
    , Logger(MakeIntrusive<NLog::TPqSpdlogWrapper>(config.GetLogName()))
{
    Y_ENSURE(config.GetProducersCount() > 0, "Producers count must be > 0");

    auto producerSettings = GetProducerSettings(config, credentialsProvider);
    auto dropLogger = config.HasDropLogName() ? NLog::GetLog(config.GetDropLogName()) : NLog::TLogPtr();

    for (size_t i = 0; i < config.GetProducersCount(); ++i) {
        producerSettings.SourceId = GetRandomSource(config.GetSourceIdPrefix());

        if (config.GetBindProducersToPartitions()) {
            producerSettings.PartitionGroup = i + 1;
        }

        Producers.emplace_back(MakeHolder<TBufferedProducer>(
                *PqLib,
                producerSettings,
                static_cast<size_t>(config.GetMaxChunkSizeBytes()),
                TDuration::Seconds(config.GetMaxFlushIntervalSeconds()),
                config.GetMaxBytesInFlight() / config.GetProducersCount(),
                Logger,
                dropLogger,
                statsSettings
         ));
    }

    Start(TDuration::Seconds(config.GetStartTimeoutSec()));
}

void TProducer::Start(TDuration timeout) {
    for (auto& producer : Producers) {
        producer->Start(timeout);
    }
}

bool TProducer::TryEnqueue(size_t partitioner, TString message) {
    auto& producer = Producers[partitioner % Producers.size()];
    return producer->TryEnqueue(std::move(message));
}

bool TProducer::TryEnqueue(TString message) {
    return TryEnqueue(RandomNumber<ui64>(), std::move(message));
}

void TProducer::Stop(TDuration timeout) {
    auto deadline = timeout.ToDeadLine();

    for (auto& producer : Producers) {
        producer->Stop(deadline - TInstant::Now());
    }
}

TProducer::~TProducer() {
    Stop();
}
