#include "single_partition_producer.h"

using namespace NCrypta;
using namespace NCrypta::NPQ;

TSinglePartitionProducer::TSinglePartitionProducer(NPersQueue::TPQLib& pqLib, const NPersQueue::TProducerSettings& producerSettings, TIntrusivePtr<NPersQueue::ILogger>& logger, TStats& stats)
    : Producer(pqLib.CreateProducer(producerSettings))
    , SourceId(producerSettings.SourceId)
    , Logger(logger)
    , Stats(stats)
{
}

void TSinglePartitionProducer::Start(TDuration timeout) {
    auto future = Producer->Start();
    future.Wait(timeout);

    Y_ENSURE(future.HasValue(), "Failed create producer until timeout");

    const auto& response = future.GetValue().Response;
    AtomicSet(SeqNo, response.GetInit().GetMaxSeqNo());
}

TSinglePartitionProducer::TCommitResponseFuture TSinglePartitionProducer::Write(TString data) {
    const auto& seqNo = AtomicIncrement(SeqNo);

    Stats.Count->Add("send.messages");

    return Producer->Write(seqNo, std::move(data)).Apply([this](const auto& future) {
        const auto& response = future.GetValue().Response;

        if (response.HasError()) {
            Log(TLOG_ERR, TStringBuilder() << "Failed to commit " << response.GetAck().GetSeqNo() << " error: " << response.GetError().GetDescription(), response);
            Stats.Count->Add("commit.errors");
        } else if (response.GetAck().GetAlreadyWritten()) {
            Log(TLOG_WARNING, TStringBuilder() << "Already written " << response.GetAck().GetSeqNo(), response);
            Stats.Count->Add("commit.duplicates");
        } else {
            Log(TLOG_WARNING, TStringBuilder() << "Successfully written " << response.GetAck().GetSeqNo(), response);
            Stats.Count->Add("commit.success");
        }

        return future;
    });
}

void TSinglePartitionProducer::Log(ELogPriority priority, const TString& message, const NPersQueue::WriteResponse& response) {
    Logger->Log(message, SourceId, response.GetInit().GetSessionId(), priority);
}

