#include "resolver.h"
#include <arpa/inet.h>

namespace NCrypta::NResolver {
    TResolveResult Resolve(const TString& host, int prefferedFamily) {
        in_addr v4;
        in6_addr v6;
        const auto hostname = host.c_str();

        if (inet_pton(AF_INET, hostname, &v4.s_addr)) {
            return v4;
        }

        if (inet_pton(AF_INET6, hostname, v6.s6_addr)) {
            return v6;
        }

        switch(prefferedFamily) {
            case AF_INET: {
                if (TryResolve<AF_INET>(hostname, &v4.s_addr)) {
                    return v4;
                }
                if (TryResolve<AF_INET6>(hostname, &v6.s6_addr)) {
                    return v6;
                }
                break;
            }
            case AF_INET6: {
                if (TryResolve<AF_INET6>(hostname, &v6.s6_addr)) {
                    return v6;
                }
                if (TryResolve<AF_INET>(hostname, &v4.s_addr)) {
                    return v4;
                }
                break;
            }
        }

        return {};
    }

    int TResolveResult::SelectedFamily() const {
        if (std::holds_alternative<in_addr>(*this)) {
            return AF_INET;
        }
        if (std::holds_alternative<in6_addr>(*this)) {
            return AF_INET6;
        }
        if (std::holds_alternative<std::monostate>(*this)) {
            return AF_UNSPEC;
        }
        Y_VERIFY(false);
    }

    const TString TResolveResult::NotResolved = "NOT RESOLVED";
}

TString ToString(const in_addr& addr) {
    return NCrypta::NResolver::ToString<AF_INET, in_addr>(addr);
}

TString ToString(const in6_addr& addr) {
    return NCrypta::NResolver::ToString<AF_INET6, in6_addr>(addr);
}

TString ToString(const NCrypta::NResolver::TResolveResult& rr) {
    switch(rr.SelectedFamily()) {
        case AF_INET:
            return ToString(std::get<in_addr>(rr));
        case AF_INET6:
            return ToString(std::get<in6_addr>(rr));
        case AF_UNSPEC:
        default:
            return NCrypta::NResolver::TResolveResult::NotResolved;
    }
}
