#include "uatraits_parser.h"

#include <contrib/libs/libarchive/libarchive/archive.h>
#include <contrib/libs/libarchive/libarchive/archive_entry.h>
#include <library/cpp/resource/resource.h>

#include <util/generic/buffer.h>
#include <util/generic/hash.h>
#include <util/generic/yexception.h>

using namespace NCrypta::NResourceService;

namespace {
    void CheckStatusIsOk(int status, const TString& errMsg) {
        Y_ENSURE(status == ARCHIVE_OK, errMsg);
    }

    struct TBuffers {
        TBuffer Browser;
        TBuffer Extra;
        TBuffer Profiles;
    };

    TBuffer Read(archive* archive, archive_entry* entry) {
        TBuffer result;
        const auto& size = archive_entry_size(entry);
        result.Resize(size);
        Y_ENSURE(size == archive_read_data(archive, result.Data(), size), "Failed to read data");
        return result;
    }
}

THolder<uatraits::detector> TUatraitsParser::Parse(const TString& input) {
    auto* uatraitsArchive = archive_read_new();
    Y_ENSURE(uatraitsArchive, "Failed to allocate archive");

    CheckStatusIsOk(archive_read_support_filter_gzip(uatraitsArchive), "Failed to set gzip");
    CheckStatusIsOk(archive_read_support_format_tar(uatraitsArchive), "Failed to set tar");
    CheckStatusIsOk(archive_read_open_memory(uatraitsArchive, input.data(), input.size()), "Failed to open archive from memory");

    archive_entry* entry;
    TBuffers buffers;

    while (archive_read_next_header(uatraitsArchive, &entry) == ARCHIVE_OK) {
        TStringBuf pathname = archive_entry_pathname(entry);
        if (pathname == "browser.xml") {
            buffers.Browser = Read(uatraitsArchive, entry);
        } else if (pathname == "extra.xml") {
            buffers.Extra = Read(uatraitsArchive, entry);
        } else if (pathname == "profiles.xml") {
            buffers.Profiles = Read(uatraitsArchive, entry);
        }
    }

    auto result = MakeHolder<uatraits::detector>(
        buffers.Browser.data(), static_cast<int>(buffers.Browser.size()),
        buffers.Profiles.data(), static_cast<int>(buffers.Profiles.size()),
        buffers.Extra.data(), static_cast<int>(buffers.Extra.size())
    );

    CheckStatusIsOk(archive_read_close(uatraitsArchive), "Failed to close archive");
    CheckStatusIsOk(archive_read_free(uatraitsArchive), "Failed to free archive");

    return result;
};

THolder<uatraits::detector> NCrypta::NResourceService::MakeDefaultUatraitsDetector() {
    return TUatraitsParser::Parse(NResource::Find("uatraits_data.tar.gz"));
}
