#include "updater.h"

#include <util/stream/str.h>

using namespace NCrypta::NResourceService;

const TString TUpdater::JugglerOkDescription = "OK";

void TUpdater::Update() {
    TGuard lock(Mutex);
    for (auto& updatable: Updatables) {
        UpdateResource(updatable);
    }
}

void TUpdater::UpdateResource(IUpdatable& updateable) {
    const auto& meta = updateable.GetResourceMeta();
    const TString service = ServiceNameBuilder(meta.Name);

    auto updateFunc = [&]() {
        const auto& remoteVersion = Client->GetVersion(meta.Name);

        if (remoteVersion != meta.Version) {
            const auto& raw = Client->GetResource(meta.Name, remoteVersion);
            updateable.Update(raw, remoteVersion);
        }
    };

    try {
        DoWithRetry<std::exception>(updateFunc, RetryOptions, true);
    } catch (const std::exception& e) {
        auto reportFailure = [&](){
            JugglerClient->ReportCrit(Host, service, e.what());
        };
        DoWithRetry<std::exception>(reportFailure, RetryOptions, false);
        return;
    }

    auto reportOk = [&]() {
        JugglerClient->ReportOk(Host, service, JugglerOkDescription);
    };
    DoWithRetry<std::exception>(reportOk, RetryOptions, false);
}

void TUpdater::Start() {
    Runner.Start();
}

TUpdater::TServiceNameBuilder TUpdater::GetDefaultServiceNameBuilder() {
    return [](const TString& resourceName) {
        return "resource/" + resourceName;
    };
}
