#include "retargeting_id.h"

#include <util/generic/hash.h>

using namespace NCrypta::NRetargetingIds;

namespace {
    constexpr ui64 METRIKA_GOAL_LOWER_BOUND = 100;
    constexpr ui64 METRIKA_GOAL_UPPER_BOUND = 1'000'000'000;
    constexpr ui64 METRIKA_SEGMENT_UPPER_BOUND = 1'500'000'000;
    constexpr ui64 LAL_SEGMENT_UPPER_BOUND = 1'900'000'000;
    constexpr ui64 MOBILE_EVENT_UPPER_BOUND = 2'000'000'000;
    constexpr ui64 METRIKA_AUDIENCE_UPPER_BOUND = 2'499'000'000;
    constexpr ui64 CRYPTA_SOCIAL_DEMO_UPPER_BOUND = 2'499'000'100;
    constexpr ui64 CRYPTA_FAMILY_UPPER_BOUND = 2'499'000'200;
    constexpr ui64 CRYPTA_BEHAVIORS_UPPER_BOUND = 2'499'001'100;
    constexpr ui64 CRYPTA_INTERESTS_UPPER_BOUND = 2'499'980'000;
    constexpr ui64 CRYPTA_INTERNAL_UPPER_BOUND = 2'499'990'000;
    constexpr ui64 MUSIC_AUDIO_GENRES_UPPER_BOUND = 2'500'000'000;
    constexpr ui64 AB_SEGMENT_UPPER_BOUND = 2'600'000'000;
    constexpr ui64 CDP_SEGMENT_UPPER_BOUND = 3'000'000'000;
    constexpr ui64 METRIKA_ECOMMERCE_UPPER_BOUND = 3'900'000'000;
    constexpr ui64 METRIKA_COUNTER_LOWER_BOUND = 4'000'000'000;
    constexpr ui64 METRIKA_COUNTER_UPPER_BOUND = 4'100'000'000;
    constexpr ui64 BRANDSAFETY_LOWER_BOUND = 4'294'967'296; // 2^32
    constexpr ui64 BRANDSAFETY_UPPER_BOUND = BRANDSAFETY_LOWER_BOUND + 1'000;
    constexpr ui64 CONTENT_CATEGORY_UPPER_BOUND = BRANDSAFETY_LOWER_BOUND + 3'000;
    constexpr ui64 CONTENT_GENRE_UPPER_BOUND = BRANDSAFETY_LOWER_BOUND + 5'000;

    static const THashMap<EType, ui64> Shifts = {
        {EType::MetrikaAudienceSegment, MOBILE_EVENT_UPPER_BOUND},
        {EType::CdpSegment, AB_SEGMENT_UPPER_BOUND},
        {EType::MetrikaEcommerce, CDP_SEGMENT_UPPER_BOUND},
        {EType::MetrikaCounter, METRIKA_COUNTER_LOWER_BOUND}
    };

    EType GetType(ui64 id) {
        if (id < METRIKA_GOAL_LOWER_BOUND) {
            return EType::SpecialGoal;
        } else if (id < METRIKA_GOAL_UPPER_BOUND) {
            return EType::MetrikaGoal;
        } else if (id < METRIKA_SEGMENT_UPPER_BOUND) {
            return EType::MetrikaSegment;
        } else if (id < LAL_SEGMENT_UPPER_BOUND) {
            return EType::DirectLalSegment;
        } else if (id < MOBILE_EVENT_UPPER_BOUND) {
            return EType::MobileEvent;
        } else if (id < METRIKA_AUDIENCE_UPPER_BOUND) {
            return EType::MetrikaAudienceSegment;
        } else if (id < CRYPTA_SOCIAL_DEMO_UPPER_BOUND) {
            return EType::CryptaSocdem;
        } else if (id < CRYPTA_FAMILY_UPPER_BOUND) {
            return EType::CryptaFamily;
        } else if (id < CRYPTA_BEHAVIORS_UPPER_BOUND) {
            return EType::CryptaBehavior;
        } else if (id < CRYPTA_INTERESTS_UPPER_BOUND) {
            return EType::CryptaInterest;
        } else if (id < CRYPTA_INTERNAL_UPPER_BOUND) {
            return EType::CryptaInternal;
        } else if (id < MUSIC_AUDIO_GENRES_UPPER_BOUND) {
            return EType::MusicAudioGenre;
        } else if (id < AB_SEGMENT_UPPER_BOUND) {
            return EType::AbSegment;
        } else if (id < CDP_SEGMENT_UPPER_BOUND) {
            return EType::CdpSegment;
        } else if (id < METRIKA_ECOMMERCE_UPPER_BOUND) {
            return EType::MetrikaEcommerce;
        } else if (id < METRIKA_COUNTER_LOWER_BOUND) {
            return EType::Unknown;
        } else if (id < METRIKA_COUNTER_UPPER_BOUND) {
            return EType::MetrikaCounter;
        } else if (id < BRANDSAFETY_LOWER_BOUND) {
            return EType::Unknown;
        } else if (id < BRANDSAFETY_UPPER_BOUND) {
            return EType::BrandSafety;
        } else if (id < CONTENT_CATEGORY_UPPER_BOUND) {
            return EType::ContentCategory;
        } else if (id < CONTENT_GENRE_UPPER_BOUND) {
            return EType::ContentGenre;
        }

        return EType::Unknown;
    }

    ui64 GetEntityId(ui64 retargetingId) {
        const auto iter = Shifts.find(GetType(retargetingId));
        return iter != Shifts.end() ? (retargetingId - iter->second) : retargetingId;
    }

    ui64 GetRetargetingId(ui64 entityId, EType type) {
        const auto iter = Shifts.find(type);
        return iter != Shifts.end() ? (entityId + iter->second) : entityId;
    }
}


TRetargetingId::TRetargetingId(ui64 entityId, EType type)
    : EntityId(entityId)
    , Type(type)
{}

TRetargetingId::TRetargetingId(ui64 retargetingId)
    : TRetargetingId(::GetEntityId(retargetingId), ::GetType(retargetingId))
{}

ui64 TRetargetingId::GetEntityId() const {
    return EntityId;
}

EType TRetargetingId::GetType() const {
    return Type;
}

ui64 TRetargetingId::Serialize() const {
    return GetRetargetingId(EntityId, Type);
}

bool TRetargetingId::operator==(const TRetargetingId& rhs) const {
    return rhs.Type == Type && rhs.EntityId == EntityId;
}

bool TRetargetingId::operator!=(const TRetargetingId& rhs) const {
    return !(*this == rhs);
}
