#pragma once

#include "solomon_exception.h"

#include <library/cpp/http/client/client.h>
#include <library/cpp/http/client/query.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/monlib/encode/json/json.h>

#include <util/generic/hash.h>
#include <util/generic/strbuf.h>
#include <util/string/builder.h>
#include <util/string/cast.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>

namespace NCrypta {
    class TSolomonDataClient {
    public:
        TSolomonDataClient(
            const TString& schema,
            const TString& host,
            const ui32 port,
            const TString& oauthToken,
            TDuration requestTimeout,
            const TString& project);

        template <typename TStringMap>
        double GetScalarSensor(const TStringMap& labels,
                               const TString& aggregationFunction,
                               TDuration aggregationWindow,
                               TInstant timeTo = TInstant::Now()) const
        {
            NJson::TJsonValue requestDtoJson;
            requestDtoJson["program"] = GetProgram(labels, aggregationFunction);
            requestDtoJson["from"] = (timeTo - aggregationWindow).ToString();
            requestDtoJson["to"] = timeTo.ToString();

            NHttp::TFetchQuery req(
                Url,
                NHttp::TFetchOptions()
                    .SetPostData(NJson::WriteJson(requestDtoJson))
                    .SetContentType(TString(NMonitoring::NFormatContenType::JSON))
                    .SetOAuthToken(OAuthToken.empty() ? Nothing() : MakeMaybe(OAuthToken))
                    .SetTimeout(RequestTimeout));

            const auto& output = NHttp::Fetch(req);
            if (output->Code != HTTP_OK) {
                ythrow TSolomonException(output->Code) << "Non-200 code from solomon: " << output->Code << ", " << output->Data;
            }

            NJson::TJsonValue response;
            NJson::ReadJsonTree(output->Data, &response, true);
            const auto& scalar = response["scalar"];

            if (scalar.IsDouble()) {
                return scalar.GetDouble();
            } else if (scalar.IsString() && scalar.GetString() == "NaN") {
                ythrow TSolomonException(output->Code) << "Found no signals in solomon response: \n" << output->Data;
            } else {
                ythrow TSolomonException(output->Code) << "Unknown scalar format: " << output->Code << ", " << output->Data;
            }
        }

    private:
        template <typename TStringMap>
        TString GetProgram(const TStringMap& labels, const TString& aggregationFunction) const {
            TStringBuilder program;
            program << aggregationFunction << "({";

            bool first = true;
            for (const auto& [label, value]: labels) {
                if (first) {
                    first = false;
                } else {
                    program << ", ";
                }
                program << label << "='" << value << "'";
            }
            program << "})";

            return program;
        }

        const TString Url;
        const TString OAuthToken;
        const TDuration RequestTimeout;
        const TString AggregationFunction;
        const TDuration AggregationWindow;
    };
}
