#pragma once

#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/log/log.h>

#include <util/system/hostname.h>
#include <util/network/socket.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/string/subst.h>

class TGraphiteSender {
public:
    TGraphiteSender(const TString& hostPort, const TString& logName = "")
        : Timeout(TDuration::MicroSeconds(1000000))
    {
        StringSplitter(hostPort).Split(':').CollectInto(&Host, &Port);
        Init(logName);
    }

    TGraphiteSender(const TString& host, ui16 port, const TString& logName = "")
        : Host(host)
        , Port(port)
        , Timeout(TDuration::MicroSeconds(1000000))
    {
        Init(logName);
    }

    template <typename T>
    void Send(const TString& metric, const T& value) const {
        TStringBuilder stringBuilder;
        stringBuilder << "one_min." << CurrentHostFqdn << "." << metric << " " << value << " " << time(0);

        try {
            Log->info("TGraphiteSender sending data: {}", stringBuilder);

            TSocket socket(TNetworkAddress(Host, Port), Timeout);
            socket.Send(stringBuilder.data(), stringBuilder.size());
        } catch (yexception& e) {
            Log->error("TGraphiteSender error: {}", e.what());
        }
    }

private:
    void Init(const TString& logName) {
        CurrentHostFqdn = NCrypta::CanonizeGraphiteMetricNode(FQDNHostName());
        Log = logName.empty() ? NCrypta::NLog::GetLog() : NCrypta::NLog::GetLog(logName);
    }

    TString Host;
    ui16 Port = 0;
    const TDuration Timeout;
    TString CurrentHostFqdn;
    NCrypta::NLog::TLogPtr Log;
};
