#pragma once

#include "map_stat.h"
#include "multi_stat.h"
#include "stat_lock_policy.h"

#include <util/generic/vector.h>
#include <util/string/cast.h>
#include <util/system/spinlock.h>

#include <cmath>

template <typename TLockPolicy = TStatLockingPolicy>
class THistStat: public IMapStat {
    template <typename T>
    friend class THistStat;

public:
    THistStat(const TString& name, double min, double max, size_t binCount)
        : IMapStat(name)
        , Min(min)
        , Max(max)
        , BinCount(binCount + 2) // two extra bins for "< min" and "> max"
        , TailMax(max)
    {
        if (max <= min) {
            ythrow yexception() << "Max <= min in stat " << name;
        }
        if (binCount < 1) {
            ythrow yexception() << "At least 1 bin required for stat " << name;
        }

        Values.resize(this->BinCount);
    }

    void Add(double value, size_t count = 1) {
        auto guard = LockPolicy.Guard();

        if (value <= Min) {
            Values[0] += count;
        } else if (value > Max) {
            Values[BinCount - 1] += count;
            TailMax = std::max(TailMax, value);
        } else {
            size_t bin = static_cast<size_t>(std::ceil((value - Min) * (BinCount - 2) / (Max - Min)));
            Values[bin] += count;
        }
    }

    template <typename T>
    void Add(const THistStat<T>& other) {
        auto guard = LockPolicy.Guard();

        if (this->Min != other.Min || this->Max != other.Max || this->BinCount != other.BinCount) {
            ythrow yexception() << "Min, Max, and BinCount must be the same: this ("
                                << this->Min << ", " << this->Max << ", " << this->BinCount << ") VS other ("
                                << other.Min << ", " << other.Max << ", " << other.BinCount << ")";
        }

        std::transform(other.Values.cbegin(), other.Values.cend(), this->Values.cbegin(), this->Values.begin(),
                       [](const size_t& countOther, const size_t& countThis) { return countOther + countThis; });
        this->TailMax = std::max(this->TailMax, other.TailMax);
    }

    void Reset() {
        auto guard = LockPolicy.Guard();

        ResetNonLocking();
    }

    TMapType GetAndReset() override {
        auto guard = LockPolicy.Guard();

        IMapStat::TMapType result;

        for (size_t i = 0; i < Values.size(); i++) {
            result[GetKey(i)] = static_cast<double>(Values[i]);
        }

        ResetNonLocking();

        return result;
    }

protected:
    double Min = 0;
    double Max = 0;
    size_t BinCount = 0;
    TVector<size_t> Values;
    double TailMax = std::numeric_limits<double>::min();

    TLockPolicy LockPolicy;

    void ResetNonLocking() {
        std::fill(Values.begin(), Values.end(), 0);
        TailMax = Max;
    }

    TString GetKey(size_t value) const {
        static const TString BIN_NAME_PREFIX = "b";

        return BIN_NAME_PREFIX + ToString(value);
    }
};

using TMultiHistStat = TMultiStat<THistStat<>, double, double, size_t>;
