#pragma once

#include "single_stat.h"
#include "map_stat.h"

#include <util/generic/hash.h>
#include <util/system/spinlock.h>

#include <tuple>

template <typename T, typename... TArgs>
class TMultiStat: public IMapStat {
    template <typename TS>
    using TStatsMap = THashMap<TString, TS>;

public:
    TMultiStat(const TString& name, const TArgs&... args)
        : IMapStat(name)
        , ConstructorArgs(args...)
    {
    }

    template <typename... TAddArgs>
    void Add(const TString& key, TAddArgs&&... args) {
        auto guard = Guard(Lock);

        auto it = Stats.find(key);
        T* stat = (it != Stats.end()) ? &it->second
                                      : &Stats.emplace(std::piecewise_construct,
                                                       std::forward_as_tuple(key),
                                                       std::tuple_cat(std::make_tuple(key), ConstructorArgs)).first->second;
        stat->Add(std::forward<TAddArgs>(args)...);
    }

    TMapType GetAndReset() override {
        TStatsMap<T> snapshot;
        {
            auto guard = Guard(Lock);
            Stats.swap(snapshot);
        }
        return GetAndReset(snapshot);
    }

private:
    template <typename TS>
    static typename std::enable_if_t<std::is_base_of<IMapStat, TS>::value, TMapType> GetAndReset(TStatsMap<TS>& stats) {
        TMapType result;

        for (auto& [statKey, stat] : stats) {
            const auto& statSnapshot = stat.GetAndReset();
            for (const auto& [key, value] : statSnapshot) {
                result.emplace(statKey + "." + key, value);
            }
        }

        return result;
    }

    template <typename TS>
    static typename std::enable_if_t<std::is_base_of<ISingleStat, TS>::value, TMapType> GetAndReset(TStatsMap<TS>& stats) {
        TMapType result;

        for (auto& [key, stat] : stats) {
            double value = stat.GetAndReset();
            result.emplace(key, value);
        }

        return result;
    }

protected:
    std::tuple<TArgs...> ConstructorArgs;
    TStatsMap<T> Stats;

    TAdaptiveLock Lock;
};
