#pragma once

#include "map_stat.h"
#include "multi_stat.h"
#include "stat_lock_policy.h"

#include <library/cpp/histogram/hdr/histogram.h>

#include <util/string/cast.h>
#include <util/string/subst.h>
#include <util/generic/vector.h>


template <typename TLockPolicy = TStatLockingPolicy>
class TPercentileStat: public IMapStat {
public:
    TPercentileStat(const TString& name, i64 max, i32 precision, const TVector<float>& percentiles = {95, 98, 99, 100})
        : IMapStat(name)
        , Hist(max, precision)
    {
        for (const auto& p: percentiles) {
            Percentiles.push_back({p, GetKey(p)});
        }
    }

    void Add(i64 value, i64 count = 1) {
        auto guard = LockPolicy.Guard();

        Hist.RecordValues(value, count);
    }

    void Add(const TPercentileStat<TStatNonLockingPolicy>& other) {
        auto guard = LockPolicy.Guard();

        Hist.Add(other.Hist);
    }

    void Reset() {
        auto guard = LockPolicy.Guard();

        Hist.Reset();
    }

    IMapStat::TMapType GetAndReset() override {
        auto guard = LockPolicy.Guard();

        IMapStat::TMapType result;

        if (Hist.GetTotalCount() == 0) {
            return result;
        }

        for (const auto& percentile : Percentiles) {
            result[percentile.Key] = Hist.GetValueAtPercentile(percentile.Value);
        }

        Hist.Reset();

        return result;
    }

private:
    template <typename TOtherLockPolicy>
    friend class TPercentileStat;

    struct TPercentile {
        float Value;
        TString Key;
    };

    static TString GetKey(float value) {
        static const TString PERCENTILE_NAME_PREFIX = "p";

        TString valueStr = ToString(value);
        SubstGlobal(valueStr, '.', '_');
        return PERCENTILE_NAME_PREFIX + valueStr;
    }

    TVector<TPercentile> Percentiles;
    TLockPolicy LockPolicy;
    NHdr::THistogram Hist;
};

using TMultiPercentileStat = TMultiStat<TPercentileStat<TStatLockingPolicy>, i64, i32, const TVector<float>>;
