#include "stats.h"

#include <util/generic/singleton.h>

THashMap<TString, TStats> TStats::Stats;
THolder<TStats> TStats::DefaultStats;
TRWMutex TStats::Lock;

TStats::TStats(const TString& name, const TSettings& settings, TStatsRegistry* statsRegistry) {
    this->Count = statsRegistry->Create<TMultiStat<TCountStat>>(name);
    this->Max = statsRegistry->Create<TMultiStat<TMaxStat>>(name);
    this->Avg = statsRegistry->Create<TMultiStat<TAvgStat>>(name);
    this->Percentile = statsRegistry->Create<TMultiPercentileStat>(name, settings.PercentMax, settings.PercentPrecision, settings.Percentiles);
    this->Hist = statsRegistry->Create<TMultiHistStat>(name, settings.HistMin, settings.HistMax, settings.HistBinCount);
}

void TStats::Register(const TString& name, const TSettings& settings) {
    TWriteGuard guard(Lock);

    Stats.emplace(std::piecewise_construct, std::forward_as_tuple(name), std::forward_as_tuple(name, settings));
}

void TStats::RegisterDefault(const TString& name, const TSettings& settings) {
    TWriteGuard guard(Lock);

    DefaultStats = MakeHolder<TStats>(name, settings);
}

void TStats::Unregister(const TString& name) {
    TWriteGuard guard(Lock);

    Stats.erase(name);
}

void TStats::UnregisterDefault() {
    TWriteGuard guard(Lock);

    DefaultStats.Destroy();
}

void TStats::UnregisterAll() {
    TWriteGuard guard(Lock);

    DefaultStats.Destroy();
    Stats.clear();
}

TStats& TStats::Get(const TString& name) {
    TReadGuard guard(Lock);

    return Stats.at(name);
}

TStats& TStats::GetDefault() {
    TReadGuard guard(Lock);

    Y_ENSURE(DefaultStats, "No default stats registered");

    return *DefaultStats;
}
