#include "string_archive.h"

#include <contrib/libs/libarchive/libarchive/archive.h>
#include <contrib/libs/libarchive/libarchive/archive_entry.h>

#include <util/generic/yexception.h>
#include <util/generic/buffer.h>

namespace {
    using NCrypta::NStringArchive::TFileInfo;

    void CheckStatusIsOk(int status, const TString& errMsg) {
        Y_ENSURE(status == ARCHIVE_OK, errMsg);
    }

    void AddEntry(struct archive* archiveDesc, const TFileInfo& fileInfo) {
        auto entry = archive_entry_new();
        Y_ENSURE(entry != nullptr, "Failed to allocate entry");

        archive_entry_set_pathname(entry, fileInfo.Name.c_str());
        archive_entry_set_filetype(entry, AE_IFREG);
        archive_entry_set_perm(entry, 0444);
        archive_entry_set_size(entry, fileInfo.Data.size());
        CheckStatusIsOk(archive_write_header(archiveDesc, entry), "Failed to write entry header");
        Y_ENSURE(archive_write_data(archiveDesc, fileInfo.Data.data(), fileInfo.Data.size()) == static_cast<long>(fileInfo.Data.size()),
                 "Failed to write entry data");
        archive_entry_free(entry);
    }

    TFileInfo ReadEntry(archive* archive, archive_entry* entry) {
        TFileInfo result;
        TBuffer buffer;

        const auto& size = archive_entry_size(entry);
        buffer.Resize(size);
        Y_ENSURE(size == archive_read_data(archive, buffer.Data(), size), "Failed to read data");

        result.Name = TString{archive_entry_pathname(entry)};
        buffer.AsString(result.Data);

        return result;
    }
}

namespace NCrypta::NStringArchive {
    TString Archive(const TVector<TFileInfo>& files, size_t bufferSize) {
        TString archiveData{bufferSize, '\0'};
        size_t used;

        auto archiveDesc = archive_write_new();
        Y_ENSURE(archiveDesc != nullptr, "Failed to allocate archive");

        CheckStatusIsOk(archive_write_add_filter_gzip(archiveDesc), "Failed to set gzip");
        CheckStatusIsOk(archive_write_set_format_v7tar(archiveDesc), "Failed to set tar");
        CheckStatusIsOk(archive_write_open_memory(archiveDesc, (void*)(archiveData.data()), bufferSize, &used), "Failed to open archive from memory");

        for (const auto& file : files) {
            AddEntry(archiveDesc, file);
        }

        CheckStatusIsOk(archive_write_close(archiveDesc), "Failed to close archive");
        CheckStatusIsOk(archive_write_free(archiveDesc), "Failed to free archive");

        archiveData.resize(used);
        return archiveData;
    }

    TVector<TFileInfo> Unarchive(const TString& rawData) {
        auto* archive = archive_read_new();
        Y_ENSURE(archive, "Failed to allocate archive");

        CheckStatusIsOk(archive_read_support_filter_gzip(archive), "Failed to set gzip");
        CheckStatusIsOk(archive_read_support_format_tar(archive), "Failed to set tar");
        CheckStatusIsOk(archive_read_open_memory(archive, rawData.data(), rawData.size()), "Failed to open archive from memory");

        archive_entry* entry;
        TVector<TFileInfo> result;
        while (archive_read_next_header(archive, &entry) == ARCHIVE_OK) {
            result.push_back(std::move(ReadEntry(archive, entry)));
        }

        CheckStatusIsOk(archive_read_close(archive), "Failed to close archive");
        CheckStatusIsOk(archive_read_free(archive), "Failed to free archive");

        return result;
    }
}
