#pragma once

#include <mapreduce/yt/interface/operation.h>
#include <library/cpp/dot_product/dot_product.h>
#include <util/generic/vector.h>
#include <util/stream/str.h>
#include <string>
#include <iostream>
#include <sstream>
#include <utility>
#include <list>
#include <cmath>
#include <ctime>
#include <cstdlib>

using namespace NYT;

namespace NVectorOperations {
    namespace {
        template <typename T>
        double Dot(const TVector<T>& first, const TVector<T>& second) {
            return DotProduct(first.begin(), second.begin(), second.size());
        }

        template <typename T>
        T ComputeNorm(const TVector<T>& vector) {
            return sqrt(Dot(vector, vector));
        }

        template <typename T>
        void Normalized(TVector<T>& vector) {
            double norm = ComputeNorm(vector);
            for (auto& value : vector) {
                value /= norm;
            }
        }

        template <typename T>
        TVector<T> Unpack(const TString& binVector, bool normalized = false) {
            size_t size = binVector.size() / sizeof(T);
            TVector<T> vector(size);
            std::memcpy(vector.begin(), binVector.data(), binVector.size());
            if (normalized) {
                Normalized(vector);
            }
            return vector;
        }

        template <typename T>
        TString Pack(const TVector<T>& vector) {
            TString result;
            result.assign(reinterpret_cast<const char*>(&vector[0]), vector.size() * sizeof(T));
            return result;
        }

        namespace NBarWithMetric {
            const TString BAR_KEY = "vector_b";
            const TString METRIC_KEY = "vector_m";

            TVector<float> GetFullVector(const TString& binVectorBar, const TString& binVectorMetric) {
                auto vector = Unpack<float>(binVectorBar, true);
                auto vectorSecond = Unpack<float>(binVectorMetric, true);
                vector.insert(vector.end(), vectorSecond.begin(), vectorSecond.end());
                Normalized(vector);
                return vector;
            }

            TVector<float> ExtractFromRow(const TNode& row) {
                return GetFullVector(row[BAR_KEY].AsString(), row[METRIC_KEY].AsString());
            }

            template <typename Vectors>
            TVector<float> Extract(Vectors& vectors) {
                return GetFullVector(vectors[BAR_KEY], vectors[METRIC_KEY]);
            }

        } // namespace NBarWithMetric

        auto ExtractFromRow = NBarWithMetric::ExtractFromRow;
    }
} // namespace NVectorOperations

template <typename Output>
class TOutputStorage {
public:
    TOutputStorage() {
        Remainder = 0.0;
        Probability = 1.0;
        Rate = 1.0;
    }

    void SetProbability(double probability) {
        Probability = probability;
    }

    void SetRate(double rate) {
        Rate = rate;
    }

    double GetProbability() {
        return Probability;
    }

    void Add(double score, Output* output) {
        Scores.push_back(std::make_pair(score, output));
    }

    size_t GetSize() {
        return Scores.size();
    }

    template <class Callback>
    void YieldTop(Callback toOutput, double correction = 1.) {
        int scoresSize = static_cast<int>(Scores.size());
        double doubleSize = Probability * correction * Rate * static_cast<double>(scoresSize);
        int size = static_cast<int>(doubleSize);
        size = std::min(scoresSize, size);
        Remainder += doubleSize - size;
        if (Remainder > 0 && size < scoresSize) {
            size++;
            Remainder--;
        }
        auto comparator = [](
            const auto& first,
            const auto& second) -> bool { return first.first > second.first; };
        std::partial_sort(Scores.begin(), Scores.begin() + size, Scores.end(), comparator);
        Scores.resize(size);
        for (auto& score : Scores) {
            toOutput(score);
        }
        Clear();
    }

    void Clear() {
        Scores.clear();
    }

private:
    double Probability;
    double Remainder;
    double Rate;
    TVector<std::pair<double, Output*>> Scores;
};
