#include "yaml2json.h"

#include <library/cpp/yaml/as/tstring.h>

namespace NCrypta {
    NJson::TJsonValue Yaml2Json(const YAML::Node& yaml, bool isRoot) {
        Y_ENSURE(!isRoot || yaml.IsMap(), "YAML root is expected to be a map");

        NJson::TJsonValue json;

        if (yaml.IsMap()) {
            for (const auto& it : yaml) {
                const auto& key = it.first.as<TString>();

                Y_ENSURE(!json.Has(key), "Duplicate key entry: " << key);

                json[key] = Yaml2Json(it.second, false);
            }
            return json;
        } else if (yaml.IsSequence()) {
            json.SetType(NJson::EJsonValueType::JSON_ARRAY);
            for (const auto& it : yaml) {
                json.AppendValue(Yaml2Json(it, false));
            }
            return json;
        } else if (yaml.IsScalar() && yaml.Tag() == "!") { // Enforce string type for quoted values
            TString _string;
            Y_ENSURE(YAML::convert<decltype(_string)>::decode(yaml, _string));
            json.SetType(NJson::EJsonValueType::JSON_STRING);
            json.SetValue(_string);
            return json;
        } else if (yaml.IsScalar()) {
            unsigned long long unsigned_long_long;
            if (YAML::convert<decltype(unsigned_long_long)>::decode(yaml, unsigned_long_long)) {
                json.SetType(NJson::EJsonValueType::JSON_UINTEGER);
                json.SetValue(unsigned_long_long);
                return json;
            }
            long long long_long;
            if (YAML::convert<decltype(long_long)>::decode(yaml, long_long)) {
                json.SetType(NJson::EJsonValueType::JSON_INTEGER);
                json.SetValue(long_long);
                return json;
            }
            bool _bool;
            if (YAML::convert<decltype(_bool)>::decode(yaml, _bool)) {
                json.SetType(NJson::EJsonValueType::JSON_BOOLEAN);
                json.SetValue(_bool);
                return json;
            }
            double _double;
            if (YAML::convert<double>::decode(yaml, _double)) {
                json.SetType(NJson::EJsonValueType::JSON_DOUBLE);
                json.SetValue(_double);
                return json;
            }
            TString _string;
            if (YAML::convert<decltype(_string)>::decode(yaml, _string)) {
                json.SetType(NJson::EJsonValueType::JSON_STRING);
                json.SetValue(_string);
                return json;
            }
        } else if (yaml.IsNull()) {
            json.SetType(NJson::EJsonValueType::JSON_NULL);
            return json;
        } else if (!yaml.IsDefined()) {
            json.SetType(NJson::EJsonValueType::JSON_UNDEFINED);
            return json;
        }

        ythrow yexception() << "Unknown type of YAML node: '" << yaml.as<TString>() << "'";
    }
}
