#include "serialize.h"

#include <library/cpp/yaml/as/tstring.h>

using namespace YAML;

namespace {
    template <typename T, typename TCallback>
    bool TryConvert(const YAML::Node& yaml, TCallback callback) {
        T value;
        bool success = YAML::convert<decltype(value)>::decode(yaml, value);
        if (success) {
            callback(value);
        }
        return success;
    }

    void FailOnUnknownType(const YAML::Node& yaml) {
        ythrow yexception() << "Unknown type of YAML node: '" << yaml.as<TString>() << "'";
    }
}

namespace NYT::NYTree {
    void Serialize(const YAML::Node& yaml, NYT::NYson::IYsonConsumer* consumer) {
        if (yaml.IsMap()) {
            consumer->OnBeginMap();
            for (const auto& it : yaml) {
                consumer->OnKeyedItem(it.first.as<TString>());
                Serialize(it.second, consumer);
            }
            consumer->OnEndMap();
        } else if (yaml.IsSequence()) {
            consumer->OnBeginList();
            for (const auto& it : yaml) {
                consumer->OnListItem();
                Serialize(it, consumer);
            }
            consumer->OnEndList();
        } else if (yaml.IsScalar()) {
            if (TryConvert<unsigned long long>(yaml, [consumer](unsigned long long x){ consumer->OnUint64Scalar(x); })) {
                // NOP
            } else if (TryConvert<long long>(yaml, [consumer](long long x){ consumer->OnInt64Scalar(x); })) {
                // NOP
            } else if (TryConvert<bool>(yaml, [consumer](bool x){ consumer->OnBooleanScalar(x); })) {
                // NOP
            } else if (TryConvert<double>(yaml, [consumer](double x){ consumer->OnDoubleScalar(x); })) {
                // NOP
            } else if (TryConvert<TString>(yaml, [consumer](const TString& x){ consumer->OnStringScalar(x); })) {
                // NOP
            } else {
                FailOnUnknownType(yaml);
            }
        } else if (yaml.IsNull()) {
            consumer->OnEntity();
        } else if (!yaml.IsDefined()) {
            ythrow yexception() << "Found undefined YAML node";
        } else {
            FailOnUnknownType(yaml);
        }
    }
}
