#include <crypta/lib/native/yaml/ytree_node/utils.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace {
    struct TTestSerializable : public NYT::NYTree::TYsonSerializable {
    public:
        TString StringField;
        ui64 Ui64Field;

        TTestSerializable() {
            RegisterParameter("string", StringField);
            RegisterParameter("ui64", Ui64Field);
        }
    };

    using TTestSerializablePtr = NYT::TIntrusivePtr<TTestSerializable>;

    const TString TEST_YAML = "string: string\n"
                              "ui64: 1234567890";
    const TString TEST_STRING = "string";
    const ui64 TEST_UI64 = 1234567890;

    void TestPositive(TTestSerializablePtr deserialized) {
        UNIT_ASSERT_EQUAL(TEST_UI64, deserialized->Ui64Field);
        UNIT_ASSERT_STRINGS_EQUAL(TEST_STRING, deserialized->StringField);
    }
} // namespace

Y_UNIT_TEST_SUITE(Utils) {
    using namespace NCrypta::NYsonUtils;

    Y_UNIT_TEST(LoadFromYamlNode) {
        auto node = YAML::Load(TEST_YAML);

        TestPositive(LoadFromYamlNode<TTestSerializable>(node));
    }

    Y_UNIT_TEST(LoadFromYamlNodeWithWrongField) {
        auto node = YAML::Load(TEST_YAML);
        node["string"] = 1;

        UNIT_ASSERT_EXCEPTION(LoadFromYamlNode<TTestSerializable>(node), NYT::TErrorException);
    }

    Y_UNIT_TEST(LoadFromYamlNodeWithMissingField) {
        auto node = YAML::Load(TEST_YAML);
        node.remove("string");

        UNIT_ASSERT_EXCEPTION(LoadFromYamlNode<TTestSerializable>(node), NYT::TErrorException);
    }

    Y_UNIT_TEST(LoadFromYamlStream) {
        TStringStream stream(TEST_YAML);

        TestPositive(LoadFromYamlStream<TTestSerializable>(stream));
    }

    Y_UNIT_TEST(LoadFromYamlEmptyStream) {
        TStringStream stream;
        UNIT_ASSERT_EXCEPTION(LoadFromYamlStream<TTestSerializable>(stream), NYT::TErrorException);
    }

    Y_UNIT_TEST(LoadFromYamlMalformedStream) {
        TStringStream stream("malformed");
        UNIT_ASSERT_EXCEPTION(LoadFromYamlStream<TTestSerializable>(stream), NYT::TErrorException);
    }

    Y_UNIT_TEST(LoadFromYamlFile) {
        const auto& path = GetOutputPath().Child("test.yson");

        TFileOutput output(path);
        output << TEST_YAML;
        output.Finish();

        TestPositive(LoadFromYamlFile<TTestSerializable>(path));
    }

    Y_UNIT_TEST(LoadFromYamlNonExistentFile) {
        UNIT_ASSERT_EXCEPTION(LoadFromYamlFile<TTestSerializable>(""), yexception);
    }

    Y_UNIT_TEST(LoadFromYamlEmptyFile) {
        const auto& path = GetOutputPath().Child("empty_test.yson");
        path.Touch();
        UNIT_ASSERT_EXCEPTION(LoadFromYamlFile<TTestSerializable>(path), NYT::TErrorException);
    }

    Y_UNIT_TEST(LoadFromYamlMalformedFile) {
        const auto& path = GetOutputPath().Child("malformed_test.yson");

        TFileOutput output(path);
        output << "malformed";
        output.Finish();

        UNIT_ASSERT_EXCEPTION(LoadFromYamlFile<TTestSerializable>(path), NYT::TErrorException);
    }
}
