#include "executer.h"

#include <library/cpp/retry/retry.h>
#include <ydb/library/yql/core/issue/yql_issue.h>

using namespace NYdb;
using namespace NYdb::NTable;

namespace {
    bool IsConstraintViolationOnly(const NYdb::TStatus& status) {
        const auto& issues = status.GetIssues();

        if (issues.Size() == 1) {
            const auto& issue = *issues.begin();
            if (issue.GetCode() == NYql::TIssuesIds::CORE_EXEC) {
                const auto& subIssues = issue.GetSubIssues();
                if (subIssues.size() == 1 && subIssues[0]->GetCode() == NYql::TIssuesIds::KIKIMR_CONSTRAINT_VIOLATION) {
                    return true;
                }
            }
        }
        return false;
    }
}

TDataQueryResult NCrypta::GetEmptyDataQueryResult(TStatus&& status) {
    return TDataQueryResult(std::move(status), {}, Nothing(), Nothing(), true, Nothing());
}

NYT::TErrorOr<TDataQueryResult> NCrypta::TryExecuteQueryWithRetries(TTableClient& tableClient, const NCrypta::TRetryableDataQuery& func) {
    auto resultPromise = NYT::NewPromise<TDataQueryResult>();

    const auto retryAsyncResult = tableClient.RetryOperation([resultPromise, func, invokerPtr = NYT::GetCurrentInvoker()](TSession session) {
        auto statusPromise = NThreading::NewPromise<TStatus>();

        invokerPtr->Invoke(BIND([func, resultPromise, statusPromise, session]() mutable {
            statusPromise.SetValue(func(session, resultPromise));
        }));

        return statusPromise.GetFuture();
    });

    auto retryResult = WaitFor(retryAsyncResult).ValueOrThrow();
    if (!retryResult.IsSuccess()) {
        return GetEmptyDataQueryResult(std::move(retryResult));
    }

    return resultPromise.ToFuture().Get();
}

TDataQueryResult NCrypta::ExecuteQueryWithRetries(TTableClient& dbClient, const NCrypta::TRetryableDataQuery& func) {
    return TryExecuteQueryWithRetries(dbClient, func).ValueOrThrow();
}

TMaybe<TDataQueryResult> NCrypta::RetryIfConstraintViolation(const std::function<TDataQueryResult()>& func) {
    auto retryFunc = [&func]() {
        const auto result = func();
        if (!result.IsSuccess() && IsConstraintViolationOnly(result)) {
            ythrow TYdbConstraintViolationException();
        }
        return result;
    };

    return DoWithRetry<TDataQueryResult, TYdbConstraintViolationException>(std::move(retryFunc), TRetryOptions(5, TDuration::Zero()), false);
}
