#pragma once

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/ydb/ydb_client.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <yt/yt/client/api/client.h>

#include <util/generic/maybe.h>

namespace NCrypta {
    NYdb::NTable::TDataQueryResult GetEmptyDataQueryResult(NYdb::TStatus&& status);

    template <class TRequest>
    TString RenderQuery(const TString& path) {
        return Sprintf(TRequest::Query, path.c_str());
    }

    template <class TRequest>
    TString RenderQuery(const TString& path, const typename TRequest::TRequestParams& requestParams) {
        return Sprintf(TRequest::GetQuery(requestParams).c_str(), path.c_str());
    }

    template <class TRequest>
    NYdb::NTable::TDataQueryResult ExecuteRenderedYdbQueryInSession(NYdb::NTable::TSession& session, const TString& query, const typename TRequest::TRequestParams& requestParams, NYdb::NTable::TTxControl&& txControl) {
        const auto& params = TRequest::GetParams(session.GetParamsBuilder(), requestParams);
        auto executeResult = WaitFor(session.ExecuteDataQuery(query, txControl, params, NYdb::NTable::TExecDataQuerySettings().KeepInQueryCache(true))).ValueOrThrow();

        if (executeResult.IsSuccess()) {
            return executeResult;
        } else {
            return GetEmptyDataQueryResult(std::move(executeResult));
        }
    }

    template<typename TRequest>
    NYdb::NTable::TAsyncDataQueryResult ExecuteRenderedYdbQuery(TYdbClient& ydbClient, const TString& query, const typename TRequest::TRequestParams& requestParams, const NYdb::NTable::TTxSettings& txSettings = NYdb::NTable::TTxSettings::SerializableRW()) {
        auto promise = NThreading::NewPromise<NYdb::NTable::TDataQueryResult>();

        auto checkDataQueryResult = [promise](NYdb::NTable::TAsyncDataQueryResult future) mutable -> NYdb::TAsyncStatus {
            auto dataQueryResult = future.GetValue();
            if (dataQueryResult.IsSuccess()) {
                promise.SetValue(dataQueryResult);
            }
            return NThreading::MakeFuture<NYdb::TStatus>(dataQueryResult);
        };

        auto operation = [query, requestParams, txSettings, checkDataQueryResult](NYdb::NTable::TSession session) mutable -> NYdb::TAsyncStatus {
            return session.ExecuteDataQuery(
                query,
                NYdb::NTable::TTxControl::BeginTx(txSettings).CommitTx(),
                TRequest::GetParams(session.GetParamsBuilder(), requestParams),
                NYdb::NTable::TExecDataQuerySettings().KeepInQueryCache(true)
            ).Apply(checkDataQueryResult);
        };

        auto checkRetryOperationResult = [promise](NYdb::TAsyncStatus future) mutable -> NYdb::NTable::TAsyncDataQueryResult {
            auto status = future.GetValue();
            if (!status.IsSuccess()) {
                promise.SetValue(GetEmptyDataQueryResult(std::move(status)));
            }
            return promise.GetFuture();
        };

        return ydbClient.TableClient.RetryOperation(operation).Apply(checkRetryOperationResult);
    }


    template<typename TRequest>
    NYdb::NTable::TAsyncDataQueryResult ExecuteYdbQuery(TYdbClient& ydbClient, const TString& relativePath, const typename TRequest::TRequestParams& requestParams, const NYdb::NTable::TTxSettings& txSettings = NYdb::NTable::TTxSettings::SerializableRW()) {
        const auto& path = ydbClient.GetAbsolutePath(relativePath);
        const auto& query = RenderQuery<TRequest>(path);

        return ExecuteRenderedYdbQuery<TRequest>(ydbClient, query, requestParams, txSettings);
    }

    template<typename TRequest>
    NYdb::NTable::TAsyncDataQueryResult ExecuteYdbDynamicQuery(TYdbClient& ydbClient, const TString& relativePath, const typename TRequest::TRequestParams& requestParams, const NYdb::NTable::TTxSettings& txSettings = NYdb::NTable::TTxSettings::SerializableRW()) {
        const auto& path = ydbClient.GetAbsolutePath(relativePath);
        const auto& query = RenderQuery<TRequest>(path, requestParams);

        return ExecuteRenderedYdbQuery<TRequest>(ydbClient, query, requestParams, txSettings);
    }

    template<typename TRequest>
    NYdb::NTable::TDataQueryResult ExecuteYdbQueryInSession(NYdb::NTable::TSession& session, const TString& path, const typename TRequest::TRequestParams& requestParams, NYdb::NTable::TTxControl&& txControl) {
        const auto& query = RenderQuery<TRequest>(path);

        return ExecuteRenderedYdbQueryInSession<TRequest>(session, query, requestParams, std::move(txControl));
    }

    template<typename TRequest>
    NYdb::NTable::TDataQueryResult ExecuteYdbDynamicQueryInSession(NYdb::NTable::TSession& session, const TString& path, const typename TRequest::TRequestParams& requestParams, NYdb::NTable::TTxControl&& txControl) {
        const auto& query = RenderQuery<TRequest>(path, requestParams);

        return ExecuteRenderedYdbQueryInSession<TRequest>(session, query, requestParams, std::move(txControl));
    }

    using TRetryableDataQuery = std::function<NYdb::TStatus(NYdb::NTable::TSession, NYT::TPromise<NYdb::NTable::TDataQueryResult>)>;

    NYT::TErrorOr<NYdb::NTable::TDataQueryResult> TryExecuteQueryWithRetries(NYdb::NTable::TTableClient& tableClient, const TRetryableDataQuery& func);
    NYdb::NTable::TDataQueryResult ExecuteQueryWithRetries(NYdb::NTable::TTableClient& tableClient, const TRetryableDataQuery& func);

    class TYdbConstraintViolationException : public yexception {};

    TMaybe<NYdb::NTable::TDataQueryResult> RetryIfConstraintViolation(const std::function<NYdb::NTable::TDataQueryResult()>& func);
}
