#include "helpers.h"

#include "join.h"
#include "ydb_error_exception.h"

#include <crypta/lib/native/concurrency/wait_for.h>

#include <library/cpp/iterator/filtering.h>

void NCrypta::ThrowOnError(const NYdb::TStatus& status, const TString& msg) {
    if (!status.IsSuccess()) {
        ythrow TYdbErrorException(status) << msg;
    }
}

TVector<NYdb::NTable::TKeyRange> NCrypta::GetTableKeyRanges(NYdb::NTable::TSession& session, const TString& path) {
    const auto settings = NYdb::NTable::TDescribeTableSettings().WithKeyShardBoundary(true);
    const auto result = WaitFor(session.DescribeTable(path, settings)).ValueOrThrow();
    ThrowOnError(result, "Can't describe table for obtaining key ranges");
    return result.GetTableDescription().GetKeyRanges();
}

bool NCrypta::IsPathExists(NYdb::NScheme::TSchemeClient& schemeClient, const TString& path) {
    const auto status = WaitFor(schemeClient.DescribePath(path)).ValueOrThrow();
    if (status.IsSuccess()) {
        return true;
    } else if (status.GetStatus() == NYdb::EStatus::SCHEME_ERROR) {
        return false;
    } else {
        ythrow TYdbErrorException(status);
    }
}

TVector<NYdb::NScheme::TSchemeEntry> NCrypta::ListDirectory(NYdb::NScheme::TSchemeClient& schemeClient, const TString& path) {
    const auto result = WaitFor(schemeClient.ListDirectory(path)).ValueOrThrow();
    ThrowOnError(result);
    return result.GetChildren();
}

void NCrypta::RemoveDirectory(NYdb::NTable::TTableClient& tableClient, NYdb::NScheme::TSchemeClient& schemeClient, const TString& path, bool recursive) {
    // TODO(kolontaev): Сделать асинхронную версию и использовать тут. Все удаления сложить в вектор и дождаться его.
    // TODO(kolontaev): В сибири вытащить дроп таблиц с ретраем сюда.
    const auto& children = ListDirectory(schemeClient, path);

    Y_ENSURE(recursive || children.empty(), "Can't remove directory with content without recursive=True");

    for (const auto& entry : children) {
        const auto& entryPath = JoinYdbPath(path, entry.Name);
        if (entry.Type == NYdb::NScheme::ESchemeEntryType::Directory) {
            RemoveDirectory(tableClient, schemeClient, entryPath, true);
        } else if (entry.Type == NYdb::NScheme::ESchemeEntryType::Table) {
            const auto asyncStatus = tableClient.RetryOperation([entryPath](NYdb::NTable::TSession session) mutable {
                return session.DropTable(entryPath);
            });
            ThrowOnError(WaitFor(asyncStatus).ValueOrThrow());
        } else {
            ythrow yexception() << "Can't remove YDB entry with type " << ToString(entry.Type);
        }
    }

    ThrowOnError(WaitFor(schemeClient.RemoveDirectory(path)).ValueOrThrow());
}

TString NCrypta::GetFreshestTablePath(TYdbClient& ydbClient, const TString& directory) {
    const auto listDirectoryResult = WaitFor(ydbClient.SchemeClient.ListDirectory(ydbClient.GetAbsolutePath(directory))).ValueOrThrow();
    ThrowOnError(listDirectoryResult);

    auto permanentTables = MakeFilteringRange(listDirectoryResult.GetChildren(), [](const auto& table) {
        return !table.Name.StartsWith(".");
    });

    const auto& it = MaxElement(permanentTables.begin(), permanentTables.end(), [&directory](const auto& lhs, const auto& rhs) {
        try {
            return FromString<ui64>(lhs.Name) < FromString<ui64>(rhs.Name);
        } catch (const TFromStringException& e) {
            ythrow yexception() << "Can't parse table names in " << directory << ". Error = " << e.what();
        }
    });

    Y_ENSURE(it != permanentTables.end(), "No permanent tables found in '" << directory << "'");
    return JoinYdbPath(directory, it->Name);
}

NYdb::TAsyncStatus NCrypta::CreateTable(TYdbClient& ydbClient, const TString& relativePath, NYdb::NTable::TTableDescription&& tableDescription, const NYdb::NTable::TCreateTableSettings& createTableSettings) {
    const auto& path = ydbClient.GetAbsolutePath(relativePath);
    return ydbClient.TableClient.RetryOperation([path, tableDescription = std::move(tableDescription), createTableSettings](NYdb::NTable::TSession session) mutable {
        return session.CreateTable(path, std::move(tableDescription), createTableSettings);
    });
}

NYdb::TAsyncStatus NCrypta::DropTable(TYdbClient& ydbClient, const TString& relativePath, const NYdb::NTable::TDropTableSettings& dropTableSettings) {
    const auto& path = ydbClient.GetAbsolutePath(relativePath);
    return ydbClient.TableClient.RetryOperation([path, dropTableSettings](NYdb::NTable::TSession session) mutable {
        return session.DropTable(path, dropTableSettings);
    });
}
