#include "kv_database.h"

#include <crypta/lib/native/yt/dyntables/kv_schema/key_rows_builder.h>
#include <crypta/lib/native/yt/dyntables/kv_schema/key_value_rows_builder.h>

#include <yt/yt/client/api/rowset.h>
#include <yt/yt/client/api/rpc_proxy/config.h>
#include <yt/yt/client/api/rpc_proxy/connection.h>

#include <utility>

using namespace NCrypta;
using namespace NCrypta::NYtDynTables;

namespace {
    NYT::NApi::IConnectionPtr CreateConnection(const TTableConfig& tableConfig) {
        auto connectionConfig = NYT::New<NYT::NApi::NRpcProxy::TConnectionConfig>();

        connectionConfig->ClusterUrl = tableConfig.GetClusterUrl();
        connectionConfig->DynamicChannelPool->MaxPeerCount = tableConfig.GetMaxPeerCount();

        if (tableConfig.HasProxyRole()) {
            connectionConfig->ProxyRole = tableConfig.GetProxyRole();
        }

        return NYT::NApi::NRpcProxy::CreateConnection(connectionConfig);
    }

    NYT::NApi::TClientOptions GetClientOptions(const TClientConfig& clientConfig) {
        NYT::NApi::TClientOptions clientOptions;
        clientOptions.Token = clientConfig.GetToken();

        return clientOptions;
    }

    NYT::NApi::TModifyRowsOptions GetModifyRowsOptions(const TModifyRowsConfig& modifyRowsConfig) {
        return {
            .RequireSyncReplica = modifyRowsConfig.GetRequireSyncReplica()
        };
    }
}

TKvAsyncDatabase::TKvAsyncDatabase(const TClientConfig& clientConfig, const TTableConfig& tableConfig, const TModifyRowsConfig& modifyRowsConfig)
        : TablePath(tableConfig.GetTablePath())
        , ModifyRowsConfig(GetModifyRowsOptions(modifyRowsConfig))
        , OperationTimeout(TDuration::MilliSeconds(tableConfig.GetOperationTimeoutMs()))
        , Client(CreateConnection(tableConfig)->CreateClient(GetClientOptions(clientConfig)))
{
}

TKvAsyncDatabase::TKvAsyncDatabase(NYT::NApi::IClientBasePtr client, const TKvAsyncDatabase& parent)
        : TablePath(parent.TablePath)
        , ModifyRowsConfig(parent.ModifyRowsConfig)
        , OperationTimeout(parent.OperationTimeout)
        , Client(std::move(client))
{
}

NYT::TFuture<TKvAsyncDatabaseTransactionPtr> TKvAsyncDatabase::StartTransaction() {
    return Client->StartTransaction(NYT::NTransactionClient::ETransactionType::Tablet, {.Timeout = OperationTimeout}).Apply(BIND([this](NYT::NApi::ITransactionPtr tx) {
        return MakeAtomicShared<TKvAsyncDatabaseTransaction>(tx, *this);
    }));
}

NYT::TFuture<TVector<TRecord>> TKvAsyncDatabase::Lookup(const TVector<TString>& keys) {
    TKeyRowsBuilder rowsBuilder;

    for (const auto& key : keys) {
        rowsBuilder.AddRow(key);
    }

    const auto& rows = rowsBuilder.Build();

    NYT::NApi::TLookupRowsOptions lookupOptions;
    lookupOptions.Timeout = OperationTimeout;

    return Client->LookupRows(TablePath, rows.NameTable, rows.Rows, lookupOptions).Apply(BIND([this](const NYT::NApi::IUnversionedRowsetPtr& rowset) { return ParseRecords(rowset); }));
}

TVector<NCrypta::TRecord> TKvAsyncDatabase::ParseRecords(const NYT::NApi::IUnversionedRowsetPtr& rowset) {
    const auto& rows = rowset->GetRows();

    TVector<TRecord> result;
    result.reserve(rows.Size());
    std::transform(rows.begin(), rows.end(), std::back_inserter(result), [this](const auto& row) { return RowParser.Parse(row); });

    return result;
}

TKvAsyncDatabaseTransaction::TKvAsyncDatabaseTransaction(const NYT::NApi::ITransactionPtr& tx, const TKvAsyncDatabase& parent)
        : TKvAsyncDatabase(tx, parent)
        , Tx(tx)
{
}

void TKvAsyncDatabaseTransaction::Write(const TVector<TRecord>& records) {
    TKeyValueRowsBuilder rowsBuilder;

    for (const auto& record : records) {
        rowsBuilder.AddRow(record.Key, record.Value);
    }

    const auto& rows = rowsBuilder.Build();

    Tx->WriteRows(TablePath, rows.NameTable, rows.Rows, ModifyRowsConfig);
}

void TKvAsyncDatabaseTransaction::Delete(const TVector<TString>& keys) {
    TKeyRowsBuilder rowsBuilder;

    for (const auto& key : keys) {
        rowsBuilder.AddRow(key);
    }

    const auto& rows = rowsBuilder.Build();

    Tx->DeleteRows(TablePath, rows.NameTable, rows.Rows, ModifyRowsConfig);
}

NYT::TFuture<NYT::NApi::TTransactionCommitResult> TKvAsyncDatabaseTransaction::Commit() {
    return Tx->Commit().WithTimeout(OperationTimeout);
}

NYT::TFuture<void> TKvAsyncDatabaseTransaction::Abort() {
    return Tx->Abort().WithTimeout(OperationTimeout);
}
