#include "kv_database.h"
#include "yt_exception.h"

#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/yt/utils/error_utils.h>

using namespace NCrypta;
using namespace NCrypta::NYtDynTables;

namespace {
    template <typename T>
    void WriteErrorCodeStats(TStats& stats, const TString& prefix, const NYT::TErrorOr<T>& error) {
        Y_ENSURE(!error.IsOK(), "Not an error");

        const auto& innerErrors = error.InnerErrors();

        if (innerErrors.empty()) {
            stats.Count->Add(prefix + ".error_code." + NErrorUtils::ToString(error.GetCode()));
            return;
        }

        for (const auto& innerError : innerErrors) {
            WriteErrorCodeStats(stats, prefix, innerError);
        }
    }
}

TKvDatabase::TKvDatabase(TKvAsyncDatabasePtr mainDatabase, TStats& stats, TVector<TKvAsyncDatabasePtr> retryDatabases, TDuration retryTimeout)
    : MainDatabase(std::move(mainDatabase))
    , RetryDatabases(std::move(retryDatabases))
    , RetryTimeout(retryTimeout)
    , Stats(stats)
{
}

TKvDatabaseTransactionPtr TKvDatabase::StartTransaction() {
    TScopeTimer scopeTimer(Stats.Percentile, "start_tx.timing");
    Stats.Count->Add("start_tx.count");

    TVector<NYT::TFuture<TKvAsyncDatabaseTransactionPtr>> futures = {
            MainDatabase->StartTransaction().Apply(BIND([=](const NYT::TErrorOr<TKvAsyncDatabaseTransactionPtr>& r) {
                if (!r.IsOK() && (r.GetCode() != NYT::EErrorCode::Canceled)) {
                    WriteErrorCodeStats(Stats, "start_tx", r);
                }

                return r;
            }))
    };

    for (auto& retryDatabase : RetryDatabases) {
        futures.push_back(
            NYT::NConcurrency::TDelayedExecutor::MakeDelayed(RetryTimeout).Apply(BIND([=]() {
                Stats.Count->Add("start_tx.retry.call");
                return retryDatabase->StartTransaction();
            })).Apply(BIND([=](const NYT::TErrorOr<TKvAsyncDatabaseTransactionPtr>& r) {
                if (r.IsOK()) {
                    Stats.Count->Add("start_tx.retry.complete");
                } else if (r.GetCode() != NYT::EErrorCode::Canceled) {
                    WriteErrorCodeStats(Stats, "start_tx", r);
                }

                return r;
            }))
        );
    }

    auto result = NYT::NConcurrency::WaitFor(NYT::AnySucceeded(std::move(futures)));

    if (result.IsOK()) {
        Stats.Count->Add("start_tx.result.success");
        return MakeAtomicShared<TKvDatabaseTransaction>(result.Value(), Stats);
    }

    Stats.Count->Add("start_tx.result.error");

    ythrow TYtException(result) << "StartTransaction failed: ";
}

TVector<TRecord> TKvDatabase::Lookup(const TVector<TString>& keys) {
    TScopeTimer scopeTimer(Stats.Percentile, "lookup.timing");
    Stats.Count->Add("lookup.count");

    TVector<NYT::TFuture<TVector<TRecord>>> futures = {
            MainDatabase->Lookup(keys).Apply(BIND([=](const NYT::TErrorOr<TVector<TRecord>>& r) {
                if (!r.IsOK() && (r.GetCode() != NYT::EErrorCode::Canceled)) {
                    WriteErrorCodeStats(Stats, "lookup", r);
                }

                return r;
            }))
    };

    for (auto& retryDatabase : RetryDatabases) {
        futures.push_back(
            NYT::NConcurrency::TDelayedExecutor::MakeDelayed(RetryTimeout).Apply(BIND([=]() {
                Stats.Count->Add("lookup.retry.call");
                return retryDatabase->Lookup(keys);
            })).Apply(BIND([=](const NYT::TErrorOr<TVector<TRecord>>& r) {
                if (r.IsOK()) {
                    Stats.Count->Add("lookup.retry.complete");
                } else if (r.GetCode() != NYT::EErrorCode::Canceled) {
                    WriteErrorCodeStats(Stats, "lookup", r);
                }

                return r;
            }))
        );
    }

    auto result = NYT::NConcurrency::WaitFor(NYT::AnySucceeded(std::move(futures)));

    if (result.IsOK()) {
        const auto& records = result.Value();

        Stats.Count->Add("lookup.result.success");
        Stats.Count->Add("lookup.records.found", records.size());
        Stats.Count->Add("lookup.records.not_found", keys.size() - records.size());

        return records;
    }

    Stats.Count->Add("lookup.result.error");

    ythrow TYtException(result) << "Lookup failed: ";
}

TKvDatabaseTransaction::TKvDatabaseTransaction(TKvAsyncDatabaseTransactionPtr asyncTx, TStats& stats)
    : TKvDatabase(asyncTx, stats)
    , AsyncTx(std::move(asyncTx))
{
}

void TKvDatabaseTransaction::Write(const TVector<TRecord>& records) {
    Stats.Count->Add("write.records", records.size());
    AsyncTx->Write(records);
}

void TKvDatabaseTransaction::Delete(const TVector<TString>& keys) {
    Stats.Count->Add("delete.records", keys.size());
    AsyncTx->Delete(keys);
}

NYT::NApi::TTransactionCommitResult TKvDatabaseTransaction::Commit() {
    TScopeTimer scopeTimer(Stats.Percentile, "commit_tx.timing");
    Stats.Count->Add("commit_tx.count");

    auto result = NYT::NConcurrency::WaitFor(AsyncTx->Commit());

    if (result.IsOK()) {
        Stats.Count->Add("commit_tx.result.success");
        return result.Value();
    }

    Stats.Count->Add("commit_tx.result.error");
    WriteErrorCodeStats(Stats, "commit_tx", result);

    ythrow TYtException(result) << "CommitTransaction failed: ";
}

void TKvDatabaseTransaction::Abort() {
    TScopeTimer scopeTimer(Stats.Percentile, "abort_tx.timing");
    Stats.Count->Add("abort_tx.count");

    auto result = NYT::NConcurrency::WaitFor(AsyncTx->Abort());

    if (result.IsOK()) {
        Stats.Count->Add("abort_tx.result.success");
        return;
    }

    Stats.Count->Add("abort_tx.result.error");
    WriteErrorCodeStats(Stats, "abort_tx", result);

    ythrow TYtException(result) << "AbortTransaction failed: ";
}
