#pragma once

#include "yaml_config_utils.h"

#include <crypta/lib/native/log/loggers/std_logger.h>
#include <crypta/lib/native/proto_secrets/remove_secrets.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/native/yt/utils/helpers.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>

#include <util/system/type_name.h>

namespace NCrypta {
    template<typename TJob, typename TJob::TConfig(*ParseConfigFunc)(int argc, const char** argv)>
    int RunJob(int argc, const char** argv) {
        NYT::Initialize(argc, argv);

        auto log = NLog::NStdLogger::RegisterLog("main", "stdout", "info");

        NYT::TConfig::Get()->Spec["annotations"]["script_name"] = TypeName<TJob>();

        try {
            Y_ENSURE(log != nullptr);

            TShiftedClock::FreezeTimestampFromEnv();
            const auto& config = ParseConfigFunc(argc, argv);

            return TJob(config, log).Do();
        } catch (const yexception& exc) {
            log->error("Error message: '{}'", exc.what());
        } catch (...) {
            log->error("Unexpected error");
        }
        return 1;
    }

    template<typename TJob, typename TJob::TConfig(*ParseConfigFunc)(const YAML::Node&)>
    int RunJob(int argc, const char** argv) {
        auto parseConfig = [](int argc, const char** argv) {
            try {
                return ParseConfigFunc(ParseYamlConfig(argc, argv));
            } catch (const YAML::Exception& exc) {
                throw yexception() << "Error while parsing yaml config. " << exc.what();
            };
        };

        return RunJob<TJob, parseConfig>(argc, argv);
    }

    template <typename TProtoConfig, int(*JobFunc)(TProtoConfig config, NLog::TLogPtr log)>
    int RunJob(int argc, const char** argv) {
        NYT::Initialize(argc, argv);

        auto log = NLog::NStdLogger::RegisterLog("main", "stdout", "info");

        char tmpbuf[1024];
        const auto& resolvedSymbol = ResolveSymbol(reinterpret_cast<void*>(JobFunc), tmpbuf, sizeof(tmpbuf));

        NYT::TConfig::Get()->Spec["annotations"]["script_name"] = TStringBuf(resolvedSymbol.Name).Before('(');

        try {
            Y_ENSURE(log != nullptr);

            TShiftedClock::FreezeTimestampFromEnv();
            const auto& config = Yaml2Proto<TProtoConfig>(ParseYamlConfig(argc, argv));

            log->info("================ Start ================");
            log->info("Config: {}", NProtoSecrets::GetCopyWithoutSecrets(config).DebugString());

            const auto result = JobFunc(std::move(config), log);

            log->info("================ Finish ================");

            return result;
        } catch (const yexception& exc) {
            log->error("Error message: '{}'", exc.what());
        } catch (...) {
            log->error("Unexpected error");
        }
        return 1;
    }
}
