#include "processed_tables_tracker.h"

#include <mapreduce/yt/util/ypath_join.h>

#include <util/generic/ymath.h>

using namespace NCrypta;

const TString TProcessedTablesTracker::SOURCE_DIR_ATTRIBUTE = "source_dir";
const TString TProcessedTablesTracker::TABLE = "table";

TProcessedTablesTracker::TProcessedTablesTracker(const TConfig& config)
    : Config(config) {
}

TProcessedTablesTracker::TProcessedTablesTracker(const TTrackedSource& trackedSource)
    : Config({ .SourceDir = trackedSource.GetSourceDir(), .StateTable = trackedSource.GetTrackTable() }) {
}

TVector<NYT::TYPath> TProcessedTablesTracker::GetUnprocessedTables(NYT::IClientBasePtr client, ui32 maxTables) {
    auto processedTables = GetProcessedTables(client);
    TVector<NYT::TYPath> unprocessedTables;

    for (const auto& path: GetAllTables(client)) {
        if (!processedTables.contains(path)) {
            unprocessedTables.emplace_back(std::move(path));
        }
    }

    Sort(unprocessedTables, [](const auto& t1, const auto& t2) { return t1 > t2; });

    if (maxTables < unprocessedTables.size()) {
        unprocessedTables.resize(maxTables);
    }

    return unprocessedTables;
}

THashSet<NYT::TYPath> TProcessedTablesTracker::GetProcessedTables(NYT::IClientBasePtr client) {
    THashSet<NYT::TYPath> processedTables;

    if (!client->Exists(Config.StateTable)) {
        return processedTables;
    }

    for (auto reader = client->CreateTableReader<NYT::TNode>(Config.StateTable); reader->IsValid(); reader->Next()) {
        processedTables.emplace(reader->GetRow().At(TABLE).AsString());
    }

    return processedTables;
}

THashSet<NYT::TYPath> TProcessedTablesTracker::GetAllTables(NYT::IClientBasePtr client) {
    auto allTables = List(client, Config.SourceDir, TListOptions().Absolute(true));
    return THashSet<NYT::TYPath>(allTables.begin(), allTables.end());
}
