#include "helpers.h"

#include <crypta/lib/native/time/shifted_clock.h>

#include <util/generic/algorithm.h>
#include <util/string/builder.h>


namespace NCrypta {
    void GetYtTables(NYT::IClientPtr client, const TString& srcDir, TVector<TString>& tables) {
        static const TString TYPE = "type";
        static const TString LINK = "link";
        static const TString MAP_NODE = "map_node";
        static const TString TABLE = "table";

        const auto& srcDirTrimmed = srcDir.EndsWith('/') ? srcDir.substr(0, srcDir.size() - 1) : srcDir;

        auto typeAttrFilter = NYT::TAttributeFilter().AddAttribute(TYPE);

        for (const auto& node : client->List(srcDirTrimmed, NYT::TListOptions().AttributeFilter(typeAttrFilter))) {
            const auto& name = node.AsString();
            const TString& nodePath = srcDirTrimmed + '/' + name;

            auto type = node.GetAttributes()[TYPE].AsString();
            if (type == LINK) {
                type = client->Get(nodePath, NYT::TGetOptions().AttributeFilter(typeAttrFilter)).GetAttributes()[TYPE].AsString();
            }

            if (type == MAP_NODE) {
                GetYtTables(client, nodePath, tables);
            } else if (type == TABLE) {
                tables.push_back(nodePath);
            }
        }
    }

    TVector<TString> GetYtTables(NYT::IClientPtr client, const TString& srcDir) {
        TVector<TString> result;
        GetYtTables(client, srcDir, result);
        return result;
    }

    TVector<TString> GetYtTables(const TString& server, const TString& srcDir) {
        auto client = NYT::CreateClient(server);
        return GetYtTables(client, srcDir);
    }

    TVector<NYT::TYPath> List(NYT::IClientBasePtr client, const NYT::TYPath& dir, const TListOptions& options) {
        TVector<NYT::TYPath> list;
        for (const auto& node: client->List(dir, options.Nyt_)) {
            const auto& name = node.AsString();
            const auto& path = options.Absolute_ ? NYT::JoinYPaths(dir, name) : name;
            list.push_back(path);
        }

        if (options.Sort_) {
            Sort(list);
        }

        return list;
    }

    void SetTtl(NYT::IClientBasePtr client, const TString& path, const TDuration& ttl, ESetTtlMode mode) {
        if ((mode == ESetTtlMode::RemoveIfEmpty) && EmptyTable(client, path)) {
            client->Remove(path);
            return;
        }

        const TInstant creationTime = TInstant::ParseIso8601(client->Get(NYT::JoinYPaths(path, "@creation_time")).AsString());
        const TInstant expirationTime = creationTime + ttl;
        SetAttribute(client, path, "expiration_time", expirationTime.ToString());
    }

    void WriteRow(NYT::TNodeWriter* writer, TMaybe<size_t> tableIndex, const NYT::TNode& row) {
        if (writer != nullptr && tableIndex.Defined()) {
            writer->AddRow(row, tableIndex.GetRef());
        }
    }

    NYT::TYPath GetPathWithAttribute(const NYT::TYPath& path, const TString& attribute) {
        return NYT::JoinYPaths(path, TStringBuilder() << "@" << attribute);
    }

    NYT::TNode GetAttribute(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& attribute) {
        return client->Get(GetPathWithAttribute(path, attribute));
    }

    bool EmptyTable(NYT::IClientBasePtr client, const NYT::TYPath& path) {
        return GetAttribute(client, path, "row_count") == 0;
    }

    void SetAttribute(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& attribute, const NYT::TNode& value) {
        client->Set(GetPathWithAttribute(path, attribute), value);
    }

    TInstant GetTimestampFromAttribute(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& attribute) {
        return TInstant::ParseIso8601(NCrypta::GetAttribute(client, path, attribute).AsString());
    }
}
