#pragma once

#include <crypta/lib/native/yql/proto_field/proto_field.h>

#include <library/cpp/threading/algorithm/parallel_algorithm.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/protos/extension.pb.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>

#include <type_traits>

namespace NCrypta {
    enum class ESetTtlMode {
        Default,
        RemoveIfEmpty
    };

    template <typename TRow, typename TContainer>
    void AddInputs(NYT::TOperationIOSpecBase& operationSpec, const TContainer& tables) {
        for (const auto& table : tables) {
            operationSpec.AddInput<TRow>(table);
        }
    };

    // TSortOperationSpec, TMergeOperationSpec
    template <typename TOperationSpec, typename TContainer>
    void AddInputs(TOperationSpec& operationSpec, const TContainer& tables) {
        for (const auto& table : tables) {
            operationSpec.AddInput(table);
        }
    }

    template <typename TRow, typename TContainer>
    void AddOutputs(NYT::TOperationIOSpecBase& operationSpec, const TContainer& tables) {
        for (const auto& table : tables) {
            operationSpec.AddOutput<TRow>(table);
        }
    };

    TVector<TString> GetYtTables(const TString& server, const TString& srcDir);
    TVector<TString> GetYtTables(NYT::IClientPtr client, const TString& srcDir);

    struct TListOptions {
        using TSelf = TListOptions;

        FLUENT_FIELD_DEFAULT(NYT::TListOptions, Nyt, NYT::TListOptions());
        FLUENT_FIELD_DEFAULT(bool, Absolute, false);
        FLUENT_FIELD_DEFAULT(bool, Sort, false);
    };

    TVector<NYT::TYPath> List(NYT::IClientBasePtr client, const NYT::TYPath& dir, const TListOptions& options = TListOptions());

    template <typename TContainer, typename TStream = IOutputStream>
    void PrintTables(const TContainer& tables, TStream& stream = Cout) {
        for (const auto& table : tables) {
            stream << table << Endl;
        }
    }

    template <typename TContainer>
    void RemoveTables(NYT::IClientBasePtr client, const TContainer& tables) {
        ParallelForEach(tables.begin(), tables.end(), [&client](const TString& path) { client->Remove(path); });
    }

    template <typename TContainer>
    TVector<NYT::TRichYPath> GetTablesWithColumns(const TContainer& tables, const TVector<TString>& columns) {
        TVector<NYT::TRichYPath> tablesWithColumns;
        for (const auto& table: tables) {
            tablesWithColumns.push_back(NYT::TRichYPath(table).Columns(columns));
        }
        return tablesWithColumns;
    }

    void SetTtl(NYT::IClientBasePtr client, const TString& path, const TDuration& ttl, ESetTtlMode mode = ESetTtlMode::Default);

    void WriteRow(NYT::TNodeWriter* writer, TMaybe<size_t> tableIndex, const NYT::TNode& row);

    NYT::TYPath GetPathWithAttribute(const NYT::TYPath& path, const TString& attribute);

    NYT::TNode GetAttribute(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& attribute);

    template <typename T>
    TMaybe<T> GetAttribute(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& attribute) {
        return client->Exists(GetPathWithAttribute(path, attribute))
               ? TMaybe<T>(GetAttribute(client, path, attribute).template As<T>())
               : Nothing();
    }

    void SetAttribute(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& attribute, const NYT::TNode& value);

    TInstant GetTimestampFromAttribute(NYT::IClientBasePtr tx, const NYT::TYPath& path, const TString& attribute);

    bool EmptyTable(NYT::IClientBasePtr client, const NYT::TYPath& path);

    template <typename T>
    void EnsureReadOnce(NYT::TNodeReader* reader, TMaybe<T>& result, std::function<T(const NYT::TNode&)> deserialize) {
        Y_ENSURE(!result.Defined());
        result = deserialize(reader->GetRow());
    }

    template<typename TProto>
    TString GetYtField(const TString& field) {
        const auto* descriptor = TProto().GetDescriptor();
        const auto* fieldDescriptor = descriptor->FindFieldByName(field);

        Y_ENSURE(fieldDescriptor != nullptr, "Proto type " << descriptor->name() << " doesn't have field " << field);

        const auto& ytField = fieldDescriptor->options().GetExtension(NYT::column_name);

        return ytField.empty() ? fieldDescriptor->name() : ytField;
    }

    template<typename TProto>
    TVector<TString> GetYtFields(const TVector<TString>& fields) {
        TVector<TString> result;
        result.reserve(fields.size());

        for (const auto& field : fields) {
            result.push_back(GetYtField<TProto>(field));
        }

        return result;
    }

    #define YT_FIELD(TProto, Field) std::is_member_function_pointer<decltype(&TProto::Get##Field)>::value ? GetYtField<TProto>(#Field) : "ERROR"

    template <typename TMessage>
    void SetYqlProtoField(NYT::IClientBasePtr client, const NYT::TYPath& path, const TString& field) {
        const auto& [attrKey, attrValue] = NYqlProtoField::GetAttr<TMessage>(field);
        SetAttribute(client, path, attrKey, attrValue);
    }

    template <typename TMessage>
    void SetYqlProtoFields(NYT::IClientBasePtr client, const NYT::TYPath& path) {
        for (const auto& [attrKey, attrValue] : NYqlProtoField::GetAttrs<TMessage>()) {
            SetAttribute(client, path, attrKey, attrValue);
        }
    }
}
