#pragma once

#include <mapreduce/yt/interface/client.h>

#include <util/generic/hash.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/ysaveload.h>

namespace NCrypta {
    template <class TAlias>
    class TTablesIndexes {
    public:
        using TMap = THashMap<TAlias, size_t>;

        TTablesIndexes() = default;

        explicit TTablesIndexes(const TMap& tablesIndexes)
            : TablesIndexes(tablesIndexes)
        {
        }

        TMaybe<size_t> operator[](const TAlias& alias) const {
            const auto& it = TablesIndexes.find(alias);
            if (it == TablesIndexes.end()) {
                return Nothing();
            }
            return it->second;
        }

        void Load(IInputStream* in) {
            TSerializer<TMap>::Load(in, TablesIndexes);
        }

        void Save(IOutputStream* out) const {
            TSerializer<TMap>::Save(out, TablesIndexes);
        }

        class TBuilder {
        public:
            using TTables = TVector<NYT::TRichYPath>;

            void Add(const NYT::TRichYPath& path, const TAlias& alias) {
                Y_ENSURE(!Indexes.contains(alias), "table with alias '" << alias << "' already added");
                Indexes[alias] = Tables.size();
                Tables.push_back(path);
            }

            template <typename TOutput, typename TSpec>
            void AddOutput(TSpec& spec, const NYT::TRichYPath& path, const TAlias& alias) {
                Add(path, alias);
                spec.template AddOutput<TOutput>(Tables.back());
            }

            template <typename TOutput, typename TSpec>
            void AddOutput(TSpec& spec, const TString& path, const TAlias& alias) {
                Add(NYT::TRichYPath(path).Schema(NYT::CreateTableSchema<TOutput>()), alias);
                spec.template AddOutput<TOutput>(Tables.back());
            }

            template <typename TInput, typename TSpec>
            void AddInput(TSpec& spec, const NYT::TRichYPath& path, const TAlias& alias) {
                Add(path, alias);
                spec.template AddInput<TInput>(Tables.back());
            }

            template <typename TInput, typename TSpec>
            void AddInput(TSpec& spec, const TString& path, const TAlias& alias) {
                AddInput<TInput, TSpec>(spec, NYT::TRichYPath(path).Schema(NYT::CreateTableSchema<TInput>()), alias);
            }

            TTablesIndexes<TAlias> GetIndexes() const {
                return TTablesIndexes(Indexes);
            }

            const TTables& GetTables() const {
                return Tables;
            }

        private:
            TMap Indexes;
            TTables Tables;
        };

    private:
        TMap TablesIndexes;
    };
}
