# -*- coding: utf-8 -*-
import abc
import json
import six.moves.urllib.parse as urlparse

import tvmauth

from crypta.lib.python.retryable_http_client import RetryableHttpClient
import crypta.lib.python.tvm.helpers as tvm


AUDIENCE_API_URL = 'https://api-audience.yandex.ru'
PORT = 443

AUDIENCE_API_TVM_ID = 2000308
AUDIENCE_INTAPI_TVM_ID = 2000306

MIN_SEGMENT_SIZE = 1000


class RetriesSettings(object):
    def __init__(self, tries_count=3, delay=3, jitter=0):
        self.tries_count = tries_count
        self.delay = delay
        self.jitter = jitter


class BaseAudienceClient(RetryableHttpClient):
    __metaclass__ = abc.ABCMeta

    @abc.abstractproperty
    def api_path_prefix(self):
        return None

    @abc.abstractproperty
    def possible_content_types(self):
        return None

    def __init__(
        self,
        oauth_token=None,
        tvm_client=None,
        tvm_dst_id=None,
        retries_settings=None,
        logger=None,
        url=AUDIENCE_API_URL,
        port=PORT
    ):
        retries_settings = retries_settings or RetriesSettings()

        super(BaseAudienceClient, self).__init__(
            retries_settings.tries_count,
            retries_settings.delay,
            retries_settings.jitter,
            timeout_in_secs=600,
            logger=logger,
        )

        self.oauth_token = oauth_token
        self.tvm_client = tvm_client
        self.tvm_dst_id = tvm_dst_id

        if not ((self.oauth_token is not None) or (self.tvm_client is not None and self.tvm_dst_id is not None)):
            raise Exception("Either oauth token or tvm client and destination tvm id should be provided")

        self.url = "{}:{}".format(url, port)
        self.logger = logger

    @classmethod
    def create_from_proto(cls, config, logger):
        def get_tvm_client(tvm_src_id, tvm_dst_id, secret):
            settings = tvmauth.TvmApiClientSettings(
                self_tvm_id=tvm_src_id,
                self_secret=secret,
                dsts=[tvm_dst_id],
                localhost_port=tvm.get_tvm_test_port(),
            )

            return tvmauth.TvmClient(settings)

        return cls(
            tvm_client=get_tvm_client(config.SrcTvmId, config.DstTvmId, config.TvmSecret),
            tvm_dst_id=config.DstTvmId,
            url=config.Url,
            port=config.Port,
            logger=logger,
        )

    @property
    def headers(self):
        if self.oauth_token is not None:
            return {'Authorization': 'OAuth {token}'.format(token=self.oauth_token)}
        else:
            return tvm.get_tvm_headers(self.tvm_client.get_service_ticket_for(tvm_id=self.tvm_dst_id))

    @staticmethod
    def _get_params(**kwargs):
        params = {k: v for k, v in kwargs.items() if v is not None}
        return params or None

    def info(self, message):
        if self.logger:
            self.logger.info(message)

    def debug(self, message):
        if self.logger:
            self.logger.debug(message)

    def list_segments(self, ulogin=None):
        return self._make_get_request(
            urlparse.urljoin(self.url, '{prefix}/segments'.format(prefix=self.api_path_prefix)),
            params=self._get_params(ulogin=ulogin),
            headers=self.headers,
        ).json()['segments']

    def get_segment_info(self, audience_segment_id, ulogin=None):
        for segment in self.list_segments(ulogin):
            if segment['id'] == audience_segment_id:
                return segment
        return None

    def update_segment_name(self, audience_segment_id, segment_name, ulogin=None):
        return self._make_put_request(
            urlparse.urljoin(
                self.url,
                '{prefix}/segment/{audience_segment_id}'.format(
                    prefix=self.api_path_prefix,
                    audience_segment_id=audience_segment_id,
                ),
            ),
            params=self._get_params(ulogin=ulogin),
            headers=self.headers,
            data=json.dumps({
                'segment': {
                    'name': segment_name,
                }
            })
        ).json()['segment']

    def upload_segment_from_data(self, data, segment_name, content_type, hashed, ulogin=None, check_size=True):
        assert content_type in self.possible_content_types

        response = self._make_post_request(
            urlparse.urljoin(self.url, '{prefix}/segments/upload_file'.format(prefix=self.api_path_prefix)),
            params=self._get_params(ulogin=ulogin),
            files={'file': data},
            headers=self.headers,
        ).json()

        self.debug(response)

        audience_segment_id = response['segment']['id']

        confirm_data = {
            'segment': {
                'id': audience_segment_id,
                'name': segment_name,
                'hashed': int(hashed),
                'content_type': content_type,
            },
        }

        confirm_response = self._make_post_request(
            urlparse.urljoin(
                self.url,
                '{prefix}/segment/{audience_segment_id}/confirm'.format(
                    prefix=self.api_path_prefix,
                    audience_segment_id=audience_segment_id,
                ),
            ),
            params=self._get_params(ulogin=ulogin, check_size=check_size),
            headers=self.headers,
            data=json.dumps(confirm_data),
        ).json()

        self.debug(confirm_response)
        self.info('Segment {} uploaded to audience'.format(audience_segment_id))

        return response['segment']

    def modify_segment_with_data(self, data, audience_segment_id, modification_type, ulogin=None, check_size=True):
        assert modification_type in ('addition', 'subtraction', 'replace')

        response = self._make_post_request(
            urlparse.urljoin(
                self.url,
                '{prefix}/segment/{audience_segment_id}/modify_data'.format(
                    prefix=self.api_path_prefix,
                    audience_segment_id=audience_segment_id,
                ),
            ),
            params=self._get_params(ulogin=ulogin, check_size=check_size, modification_type=modification_type),
            files={'file': data},
            headers=self.headers,
        ).json()

        self.debug(response)

        return response['segment']

    def upload_segment_from_file(self, filename, segment_name, content_type, hashed, ulogin=None):
        assert content_type in self.possible_content_types

        with open(filename, 'rb') as f:
            return self.upload_segment_from_data(f, segment_name, content_type, hashed, ulogin)

    def modify_segment_with_data_from_file(self, filename, audience_segment_id, modification_type, ulogin=None):
        with open(filename, 'rb') as f:
            return self.modify_segment_with_data(f, audience_segment_id, modification_type, ulogin)

    def delete_segment(self, audience_segment_id, ulogin=None):
        response = self._make_delete_request(
            urlparse.urljoin(
                self.url,
                '{prefix}/segment/{audience_segment_id}'.format(
                    prefix=self.api_path_prefix,
                    audience_segment_id=audience_segment_id,
                ),
            ),
            params=self._get_params(ulogin=ulogin),
            headers=self.headers,
        )
        return response.json()['success']


class PublicApiAudienceClient(BaseAudienceClient):
    api_path_prefix = 'v1/management'

    possible_content_types = {
        'idfa_gaid',
        'mac',
        'crm',
    }

    def list_grants(self, segment_id):
        return self._make_get_request(
            urlparse.urljoin(self.url, '{prefix}/segment/{segment_id}/grants'.format(prefix=self.api_path_prefix, segment_id=segment_id)),
            headers=self.headers,
        ).json()['grants']

    def list_grants_users(self, segment_id):
        return [x['user_login'] for x in self.list_grants(segment_id)]

    def add_grant(self, segment_id, login):
        response = self._make_put_request(
            urlparse.urljoin(self.url, '{prefix}/segment/{segment_id}/grant'.format(prefix=self.api_path_prefix, segment_id=segment_id)),
            headers=self.headers,
            data=json.dumps({'grant': {'user_login': login, 'comment': 'API'}})
        ).json()

        self.debug(response)
        self.info('Segment {} granted to {}'.format(segment_id, login))

        return response['grant']

    def delete_grant(self, segment_id, login):
        response = self._make_delete_request(
            urlparse.urljoin(self.url, '{prefix}/segment/{segment_id}/grant'.format(prefix=self.api_path_prefix, segment_id=segment_id)),
            headers=self.headers,
            params={'user_login': login},
        ).json()

        self.info('Grant for user {} deleted. Segment id = {}'.format(login, segment_id))
        return response

    def add_grants(self, segment_id, user_logins):
        for user_login in user_logins:
            self.add_grant(segment_id, user_login)

    def delete_grants(self, segment_id, user_logins):
        for user_login in user_logins:
            self.delete_grant(segment_id, user_login)

    def delete_all_segment_grants(self, segment_id):
        self.delete_grants(segment_id, self.list_grants_users(segment_id))


class PrivateApiAudienceClient(BaseAudienceClient):
    api_path_prefix = 'v1/management/client'

    possible_content_types = {
        'yuid',
    }
