import os
import sys
import types

from library.python.svn_version import svn_revision
import six

from crypta.lib.python.bt.commons.cli import shell


class NormalExit(SystemExit):
    def __init__(self, message):
        print(message)
        sys.exit(0)


def module_file(module):
    return os.path.basename(module.__file__).replace('.pyc', '.py')


def call_module_main(module_name, module, argv):
    doc = module.__doc__
    if not doc:
        raise SystemExit('Invalid module %s' % (module.__file__))
    doc = doc.replace(
        module_file(module),
        '{binary} {module}'.format(binary=shell.colorize(os.path.basename(sys.argv[0]), color=2), module=module_name),
    )
    try:
        cmd_args = shell.args(doc, argv=[module_name] + argv)
    except shell.FailedToParseArguments:
        raise SystemExit('Failed to parse arguments. Usage is: \n %s' % (doc))
    from crypta.lib.python.bt.conf.conf import ImmutableAttrDict
    return module.main(ImmutableAttrDict(cmd_args))


def get_available_tools(tools):
    queue = list(tools.items())
    visited = set()
    while queue:
        name, module = queue.pop()
        visited.add(module)
        if hasattr(module, 'main'):
            description = module.__doc__.split('Usage:')[0].rstrip()
            yield name, description or 'No description.'
        else:
            for name, value in six.iteritems(module.__dict__):
                if isinstance(value, types.ModuleType) and value not in visited:
                    queue.insert(0, value)


def file_to_module(filename):
    return filename.replace('.py', '').replace('/', '.')


def tool_example_line(path, description):
    return '     - {description}\n    {binary} {module}'.format(
        description=description,
        binary=shell.colorize(os.path.basename(sys.argv[0]), color=2),
        module=shell.colorize(file_to_module(path), color=3)
    )


def with_examples(doc, tools):
    tools = (tool for tool in get_available_tools(tools))
    examples = '\n\n'.join(tool_example_line(*tool) for tool in tools)
    return doc.replace('Examples:', 'Examples:\n\n' + examples)


def split_args(doc, argv):
    """Split arguments:
        - if -h or --help is detected, just show help
        - if --config is detected, keep 3 first arguments (--config CONFIG
          TOOL) and then the rest
        - otherwise keep just the first argument TOOL and then the rest
    """
    if len(argv) < 2:
        raise NormalExit(doc)
    if argv[1] in ('-h', '--help'):
        raise NormalExit(doc)
    if argv[1] in ('-v', '--version'):
        raise NormalExit('Version: {}'.format(svn_revision()))
    if argv[1] == '--config':
        return argv[1:4], argv[4:]
    else:
        return [argv[1]], argv[2:]


def main(doc, tools, conf_path='/'):
    doc = with_examples(doc, tools)
    argv, argv_tool = split_args(doc, sys.argv)
    try:
        args = shell.args(doc, argv=argv)
    except shell.FailedToParseArguments:
        print('Wrong arguments', argv)
        print(doc)
        sys.exit(1)

    import crypta.lib.python.bt.conf.conf as conf
    args_config = args['--config']
    if args_config:
        conf.use(args_config)
    else:
        import crypta.lib.python.bt.conf.resource_conf as resource_conf
        conf.use(resource_conf.find(conf_path))

    module_name = args['TOOL']
    if module_name not in tools:
        raise SystemExit('No such tool %s' % (module_name))

    rc = call_module_main(module_name, tools[module_name], argv_tool)
    sys.exit(rc)
