import subprocess
import sys
import logging
import os

import docopt

logger = logging.getLogger(__name__)


RESET_SEQ = "\033[0m"
COLOR_SEQ = "\033[1;%dm"
BOLD_SEQ = "\033[1m"


class FailedToParseArguments(Exception):
    pass


def run(command):
    """Calls provided shell command.
    Fails when the returncode is non zero.

    """
    logger.info('Calling shell command [%s]', command)
    process = subprocess.Popen(command, stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE, shell=True)
    out, err = process.communicate()
    if process.returncode != 0:
        logger.error('Command %s return code %d', command, process.returncode)
        raise Exception(err)
    logger.info('Stdout: %s', out)
    logger.info('Stderr: %s', err)


def var(key):
    """Return environment variable, fails otherwise.
    """
    if key in os.environ:
        return os.environ[key]
    else:
        raise Exception('Failed to get env variable %s' % (key))


def writeln(data):
    """Writes line."""
    sys.stdout.write(data)
    sys.stdout.write(os.linesep)


def colorize(text, color=1):
    """Colorizes text."""
    return '%s%s%s' % (COLOR_SEQ % (30 + color), text, RESET_SEQ)


def bold(text):
    """Bolds text."""
    return '%s%s%s' % (BOLD_SEQ, text, RESET_SEQ)


def args(docstring, argv=None):
    """Parse command arguments using docopt string.

    Uses sys.argv by default.

    :param docstring: string description of the command-line interface
    :param argv: arguments to parse, sys.argv is used by default
    """
    try:
        return docopt.docopt(docstring, argv)
    except docopt.DocoptExit as e:
        raise FailedToParseArguments('Failed to parse arguments: \n' + str(e))
