import struct

import numpy as np


def norm(vector):
    return float(np.sqrt(np.dot(vector, vector)))


def normalize(vector):
    return np.array(vector) / norm(vector)


def get_log_weighted_vector(vector):
    v_norm = norm(vector)
    weight = (1. + np.log(v_norm)) / v_norm
    return weight * vector


def cosine(first_vector, second_vector):
    if first_vector is None or second_vector is None:
        return 0.

    first_vector = np.array(first_vector)
    second_vector = np.array(second_vector)
    first_norm = norm(first_vector)
    second_norm = norm(second_vector)

    if first_vector.shape != second_vector.shape:
        raise ValueError('Computing cosine between different shaped vector')

    return float(np.sum(first_vector*second_vector) / first_norm / second_norm)


def pack(vector):
    if isinstance(vector, np.ndarray) or isinstance(vector, list):
        return struct.pack('f'*len(vector), *list(vector))
    else:
        return ""


def unpack(vector, with_log_weight=False, normalized=False):
    if isinstance(vector, str):
        vector = np.fromstring(vector, '<f4')
    if with_log_weight:
        return get_log_weighted_vector(vector)
    if normalized:
        return normalize(vector)
    return vector
