import crypta.lib.python.bt.workflow.targets.table as table_targets
import crypta.lib.python.bt.workflow.targets.directory as directory_targets

import uuid
import os.path
from collections import namedtuple

import logging
logger = logging.getLogger(__name__)


WithYtAndPath = namedtuple('WithYtAndPath', ['yt', 'path'])

WithYt = namedtuple('WithYt', ['yt'])


class PathWithYt(str):
    def __new__(cls, yt, text=None):
        if text is None:
            yt, text = None, yt
        if yt is None and isinstance(text, cls):
            yt = cls.yt
        instance = str.__new__(cls, text)
        instance.yt = yt
        instance.attributes = getattr(text, "attributes", {})

        return instance

    def get_self_with_another_yt(self, yt):
        self_with_another_yt = self.__class__(yt, self)
        self_with_another_yt.attributes = {}
        return self_with_another_yt


class YtObjectTargets(WithYtAndPath):

    def exists(self):
        return table_targets.Exists(self.yt, self.path)


class DirectoryTargets(YtObjectTargets):

    def empty(self):
        return directory_targets.Empty(self.yt, self.path)

    def exists(self):
        return table_targets.Exists(self.yt, self.path)


class TableTargets(YtObjectTargets):

    def not_empty(self):
        return table_targets.NotEmpty(self.yt, self.path)

    def not_empty_and_sorted(self):
        return table_targets.NotEmptyAndSorted(self.yt, self.path)


class YtObject(PathWithYt):
    ACCESS_TIME = 'access_time'
    CREATION_TIME = 'creation_time'
    MODIFICATION_TIME = 'modification_time'

    @property
    def be(self):
        return YtObjectTargets(yt=self.yt, path=self)

    @property
    def modification_time(self):
        return self.get_attribute(self.MODIFICATION_TIME)

    @property
    def creation_time(self):
        return self.get_attribute(self.CREATION_TIME)

    @property
    def access_time(self):
        return self.get_attribute(self.ACCESS_TIME)

    def create(self, type, recursive=True, ignore_existing=True, **kwargs):
        self.yt.create(type, self, recursive=recursive,
                       ignore_existing=ignore_existing, **kwargs)

    def remove(self, recursive=True, **kwargs):
        self.yt.remove(self, recursive=recursive, **kwargs)

    def get_attribute(self, attr_name, default=None,
                      use_cached_attr=False, replace_none_to_default=False,
                      **kwargs):
        not_none_default = 'not_none'
        if use_cached_attr:
            value = self.attributes.get(attr_name, not_none_default)
            if value != not_none_default:
                if replace_none_to_default and value is None:
                    return default
                return value
        logger.debug('%s get attribute %s', self, attr_name)
        _default = not_none_default
        value = self.yt.get_attribute(self, attr_name, _default, **kwargs)
        if value == not_none_default:
            return default
        self.attributes[attr_name] = value
        if replace_none_to_default and value is None:
            return default
        return value

    def set_attribute(self, *args, **kwargs):
        self.yt.set_attribute(self, *args, **kwargs)

    def exists(self):
        return self.yt.exists(str(self))


class Directory(YtObject):

    @property
    def be(self):
        return DirectoryTargets(yt=self.yt, path=self)

    def list(self, max_size=None, **kwargs):
        return self.yt.list(self, max_size=max_size, **kwargs)

    def _child(self, name):
        return os.path.join(self, name)

    def child_file(self, name):
        return File(self.yt, self._child(name))

    def child_table(self, name):
        return Table(self.yt, self._child(name))

    def child_directory(self, name):
        return Directory(self.yt, self._child(name))

    def create(self, **kwargs):
        super(Directory, self).create('map_node', **kwargs)

    @property
    def count(self):
        return self.get_attribute('count')


class Table(YtObject):
    ROW_COUNT = 'row_count'

    @property
    def be(self):
        return TableTargets(yt=self.yt, path=self)

    @property
    def size(self):
        return self.get_attribute(Table.ROW_COUNT)

    @property
    def foreign(self):
        return '<foreign=true>'+self

    def create(self, **kwargs):
        super(Table, self).create('table', **kwargs)

    def read(self, columns=None, *args, **kwargs):
        return self.yt.read_table(self.path(columns=columns), *args, **kwargs)

    def _limited_path(self):
        try:
            return self + self.slice
        except AttributeError:
            return self

    def path(self, **kwargs):
        return self.yt.TablePath(self._limited_path(), **kwargs)

    def limit(self, lower, upper):
        self.slice = '[#{}:#{}]'.format(lower, upper)
        return self

    def get(self, index):
        return self.yt.read_table(self + '[#{}]'.format(index))


class File(YtObject):
    def read(self, *args, **kwargs):
        return self.yt.read_file(self, *args, **kwargs)

    def write(self, *args, **kwargs):
        return self.yt.write_file(self, *args, **kwargs)

    def create(self, **kwargs):
        super(File, self).create('file', **kwargs)


class WithID(object):
    @property
    def id(self):
        return os.path.basename(self)

    def as_tuple(self):
        return self.id, self


class BasePaths(WithYt):

    @staticmethod
    def generate_new_batch_id():
        return 'batch-{}'.format(uuid.uuid4())

    def table(self, path):
        return Table(self.yt, path)

    def directory(self, path):
        return Directory(self.yt, path)
