import abc
import contextlib
import logging
import pickle
import types

from library.python.svn_version import svn_revision
import six

from crypta.lib.python.bt.workflow import exception

logger = logging.getLogger(__name__)


class TaskRegistrar(abc.ABCMeta):
    """Metaclass for registering tasks by their fully qualified name."""
    __tasks = {}
    __abstract_tasks = {}

    def __new__(self, cls, bases, classdict):
        the_class = super(TaskRegistrar, self).__new__(self, cls,
                                                       bases, classdict)
        # we register all non-abstract classes
        if not the_class.__abstractmethods__:
            self.__tasks[the_class.full_name()] = the_class
        else:
            self.__abstract_tasks[the_class.full_name()] = the_class
        return the_class

    @classmethod
    def get(self, name):
        """Returns class of task by its name.

        :param name: fully qualified name of task (e.g. somemodule.SomeTask)
        :raises: MissedTask
        """
        if name not in self.__tasks:
            if name not in self.__abstract_tasks:
                raise exception.TaskMissed(name)
            else:
                raise exception.TaskAbstract(name)
        else:
            return self.__tasks[name]

    @classmethod
    def all_tasks(self):
        return list(self.__tasks.items())


class NoDefault(object):
    pass


class Parameter(object):
    """Parameter class.
    """
    def __init__(self, parse=str, default=NoDefault()):
        self.parse = parse
        self.default = default


DEFAULT_TAG = 'default'


class Task(six.with_metaclass(TaskRegistrar, object)):
    """Base class for tasks. In this context, such a task
    can be considered as an equivalent of Make target: the task
    can be either complete or not and has a set of requirements which
    are other tasks. Task should be executed only when it is not
    complete yet *and* all its requirements are already complete.
    By default, task is considered complete when all its targets
    (:class:`Target`) are satisfied.

    This class follows design that is very similar to the Luigi library.
    """

    priority = Parameter(parse=int, default=0)

    timeout = None

    tag = Parameter(default=DEFAULT_TAG)

    class Status(object):
        ENQUEUED = 'enqueued'
        COOLING_DOWN = 'cooling down'
        LOCKED = 'locked'
        COMPLETE = 'complete'
        FAILING = 'failing'
        INVALID = 'invalid'

    def __init__(self, version=None, **kwargs):
        """Extracts all parameters (:class:`Parameter`) from class
        definition and retrieves them from kwargs.

        :raises: MissedParameter
        :raises: UnexpectedParameter
        """
        for provided_parameter_name in kwargs.keys():
            if not self._is_parameter(provided_parameter_name):
                raise exception.ParameterUnexpected(provided_parameter_name)

        for parameter_name, parameter_instance in self._parameters():
            if parameter_name not in kwargs:
                default_value = parameter_instance.default
                if not isinstance(default_value, NoDefault):
                    value = default_value
                else:
                    raise exception.ParameterMissed(parameter_name)
            else:
                value = kwargs[parameter_name]

            if value is None:
                setattr(self, parameter_name, None)
            else:
                setattr(self, parameter_name, parameter_instance.parse(value))

        self.version = version or svn_revision()

    @classmethod
    def _parameters(cls):
        for each in dir(cls):
            each_object = getattr(cls, each)
            if isinstance(each_object, Parameter):
                yield each, each_object

    @classmethod
    def _is_parameter(cls, candidate):
        try:
            attr = getattr(cls, candidate)
            return isinstance(attr, Parameter)
        except AttributeError:
            return False

    def __repr__(self):
        parameters = ('{}={}'.format(k, v)
                      for (k, v) in self._parameters_with_values())

        return '{}({})'.format(self.full_name(), ';'.join(parameters))

    def _parameters_with_values(self):
        for name, parameter in self._parameters():
            value = getattr(self, name)
            if not parameter.default == value:
                yield name, getattr(self, name)

    def __hash__(self):
        return hash(repr(self))

    def __eq__(self, other):
        return isinstance(other, self.__class__) and hash(self) == hash(other)

    def process_dependency(self, dep):
        # inherit tags
        dep.tag = self.tag
        return dep

    @abc.abstractmethod
    def requires(self):
        """Return all requirements of the task - other tasks that
        should be done first."""
        pass

    def processed_requires(self):
        for dep in self.requires():
            if dep is None:
                raise exception.TaskInvalid("Dependency %s is None" % dep)
            yield self.process_dependency(dep)

    @property
    def valid(self):
        return True

    def has_cyclic_dependencies(self):
        queue = [dep for dep in self.processed_requires()]
        while queue:
            each = queue.pop()
            if each == self:
                return True
            else:
                queue += [r for r in each.processed_requires()]
        return False

    def all_targets_satisfied(self):
        """Whether all targets are satisfied.
        If any target is not satisfied yet, True is returned. False is
        returned otherwise *but* None is returned when task has got
        no targets.
        """
        got_any = False
        for target in self.targets():
            got_any = True
            logger.debug("Checking that %s", target)
            if not target.satisfied():
                logger.info("It is not true that %s", target)
                return False
        if got_any:
            return True
        else:
            return None

    def complete(self):
        """Whether task is complete.
        Some tasks are never complete. None is returned in the case.
        """
        return self.all_targets_satisfied()

    def targets(self):
        """Return targets of task."""
        return iter(())

    @property
    def the_target(self):
        targets = list(self.targets())
        if len(targets) == 1:
            return targets[0]
        else:
            raise ValueError('Expected only one target')

    @contextlib.contextmanager
    def run_context(self):
        yield None

    @abc.abstractmethod
    def run(self, **kwargs):
        """Perform what it is needed to do."""

    def processed_run(self, **kwargs):
        results = self.run(**kwargs)
        if isinstance(results, types.GeneratorType):
            for dep in results:
                yield self.process_dependency(dep)

    @classmethod
    def full_name(cls):
        return '{}.{}'.format(cls.__module__, cls.__name__)

    @classmethod
    def short_name(cls):
        return '.'.join(
            [x[:1] for x in str(cls.__module__).split('.')] + [cls.__name__]
        )

    @staticmethod
    def dumps(task):
        return pickle.dumps(task, protocol=0)

    @staticmethod
    def loads(data):
        return pickle.loads(data)

    @classmethod
    def declaration(cls):
        fmt = '{}({})'
        parameters_desc = ','.join(name for (name, _) in cls._parameters())
        return fmt.format(cls.full_name(), parameters_desc)

    @classmethod
    def description(cls):
        return cls.__doc__ or "No description"


class SinkTask(Task):
    """Task that does nothing but considered done when all
    its requirements are complete."""
    def complete(self):
        for r in self.processed_requires():
            if not r.complete():
                return False
        return True

    def run(self):
        pass


class IndependentTask(Task):
    """Task that has no dependencies on other tasks."""
    def requires(self):
        return iter(())


class Target(six.with_metaclass(abc.ABCMeta, object)):
    """Target that could be either satisfied or not. Targets are used
    to check whether some instance :class:`Task` is already complete."""

    @abc.abstractmethod
    def satisfied(self):
        """Returns True if target is satisfied (no need to take
        any action), False otherwise."""
        pass


def available_tasks():
    """Returns list of all available tasks."""
    return TaskRegistrar.all_tasks()


def task_instance(name, **kwargs):
    """Returns task instance by its name and parameters.

    :param name: fully qualified name of the task, e.g.
                 workflow.basic.PingTask
    :param kwargs: task parameters
    """
    return TaskRegistrar.get(name)(**kwargs)


def transitive_requirements(task):
    """Returns a list of all transitive requirements of
    given task."""
    queue = [task]
    requirements = []
    while queue:
        each = queue.pop()
        requirements += [r for r in each.processed_requires()]
    return requirements
