"""Scha, a distributed module to schedule tasks execution.

Scha operates on Tasks (:class:`Task`) that are just some pieces of work to be
done. Tasks are performed by Workers (:class:`Worker`). Workers are essentially
some processes that retrieve available Tasks from the database (Zookeeper).
Before performing any Task, Workers lock them using ephemeral nodes.
An important feature is that Task execution stops immediately when connection
is lost. This approach ensures no Task is being performed by a few Workers
at the same time.

When some Worker tries to perform a Task, it stores an Attempt - a piece of
information about time of execution and it status (success or failure).
"""
import collections
import contextlib
import datetime
import functools
import hashlib
import heapq
import json
import logging
import multiprocessing
import os
import random
import signal
import sys
import time
import types
import uuid

import kazoo.client
import kazoo.exceptions
import kazoo.protocol.paths
import kazoo.recipe.lock
from library.python.svn_version import svn_revision
from setproctitle import (
    getproctitle,
    setproctitle,
)
import six
import six.moves.queue
from tblib import pickling_support

from crypta.lib.python.bt.workflow import base
from crypta.lib.python.bt.workflow.exception import (
    RecoverableError,
    NoTasksLeft,
    TaskAlreadyEnqueued,
    TaskAwaiting,
    TaskCyclic,
    TaskDisappeared,
    TaskExecuted,
    TaskFailed,
    TaskInvalid,
    TaskLocked,
    TaskNotEnqueued,
    TaskOutdated,
    TaskUncompletable,
    TransactionFailed,
    WorkerStopped,
)
from crypta.lib.python.logging import log_context

logger = logging.getLogger(__name__)


class TaskDone(object):
    """Used to indicate task was done."""


ExecutionErrorInfo = collections.namedtuple(
    'ExecutionErrorInfo', ['cls', 'error', 'traceback']
)


class Paths(object):
    """Defines how data is being stored."""

    TASKS_PATH = 'tasks'
    LOCKS_PATH = 'locks'
    ROOT_PATH = 'scha'
    WORKERS_PATH = 'workers'

    @staticmethod
    def _paths_join(*args):
        return kazoo.protocol.paths.join(*args)

    @classmethod
    def root(cls):
        return cls._paths_join('/cryptabt', cls.ROOT_PATH)

    @classmethod
    def task_id(cls, task):
        return hashlib.md5(six.ensure_binary(repr(task))).hexdigest()

    @classmethod
    def tasks(cls):
        """Returns path to all tasks."""
        return cls._paths_join(cls.root(), cls.TASKS_PATH)

    @classmethod
    def locks(cls):
        """Returns path to all locks."""
        return cls._paths_join(cls.root(), cls.LOCKS_PATH)

    @classmethod
    def lock(cls, task):
        """Returns path to lock of task."""
        return cls._paths_join(cls.locks(), cls.task_id(task))

    @classmethod
    def lock_by_task_id(cls, task_id):
        """Returns path to lock by task id."""
        return cls._paths_join(cls.locks(), task_id)

    @classmethod
    def task_by_id(cls, task_id):
        """Returns path to task with given id."""
        return cls._paths_join(cls.tasks(), task_id)

    @classmethod
    def task(cls, task):
        """Returns path to given task."""
        existing_id = getattr(task, '_id', None)
        if existing_id:
            return cls.task_by_id(existing_id)
        else:
            return cls._paths_join(cls.tasks(), cls.task_id(task))

    @classmethod
    def workers(cls):
        """Returns path to all workers."""
        return cls._paths_join(cls.root(), cls.WORKERS_PATH)

    @classmethod
    def worker(cls, worker_id):
        """Returns path to worker."""
        return cls._paths_join(cls.workers(), str(worker_id))


class TaskLayout(object):

    PRIORITY = 'Priority'
    MARSHALLED = 'Marshalled'
    TAG = 'Tag'

    def __init__(self, zk, task_id):
        self.zk = zk
        self.task_id = task_id

    @classmethod
    def store(cls, transaction, task):
        path = Paths.task(task)
        data = {cls.MARSHALLED: six.ensure_text(base.Task.dumps(task)),
                cls.PRIORITY: str(task.priority),
                cls.TAG: task.tag}
        transaction.create(path)
        transaction.set_data(path, six.ensure_binary(json.dumps(data)))

    @classmethod
    def delete(cls, transaction, task):
        path = Paths.task(task)
        logger.debug('Deleting path %s', path)
        transaction.delete(path)

    def _get(self, key, default=None):
        path = Paths.task_by_id(self.task_id)
        try:
            raw_data = self.zk.get(path)[0]
            data = json.loads(raw_data)
            if default:
                return data.get(key, default)
            else:
                return data.get(key)
        except kazoo.exceptions.NoNodeError:
            raise TaskDisappeared(self.task_id)
        except (ValueError, KeyError):
            raise TaskInvalid(self.task_id)

    def priority(self):
        return int(self._get(self.PRIORITY))

    def marshalled(self):
        return six.ensure_binary(self._get(self.MARSHALLED))

    def tag(self):
        return self._get(self.TAG, default=base.DEFAULT_TAG)


class Worker(object):
    """Performs enqueued tasks (:class:`Task`).

    Until stopped, worker's main loop glances over available tasks and
    runs ones that have no unsatisfied dependencies. Each of task's
    unsatisfied dependencies is enqueued again so that some of the workers
    can execute it and resolve the dependency.

    Worker stops task execution immediately if connection is dropped. This
    prevents tasks from multiple execution at the very same time.

    Error handling is tough in such a worker. There are a few non-critical
    errors and situations which should not stop the worker. These go
    as :class:`RecoverableError`.

    Zookeeper connection error is critical and is not handled
    as there is no guarantee the worker can recover.

    """
    class Callbacks(object):
        def __init__(self):
            def nop(_):
                pass
            self.before_acquiring_lock = nop
            self.before_executing = nop
            self.before_task_loading = nop
            self.before_receiving_result = nop
            self.before_sorting_by_priority = nop

    def __init__(self, zk, tag=None, version=None, worker_id=None, do_fork=None):
        self.zk = zk
        self.stopped = False
        self.tag = tag or base.DEFAULT_TAG
        self.lock_timeout = 0.001
        self.version = version or svn_revision()
        self.polling_interval = 1
        self.polling_jitter = 0.1
        self.callbacks = self.Callbacks()
        self.worker_id = worker_id or uuid.uuid1()
        self.do_fork = do_fork if do_fork is not None else True

    def delete(self, task, with_lock):
        """Removes a task from the queue. The task to be removed
        is locked in the first place.

        Note: If it is impossible to acquire the lock, it seems there is some
        time discrepancy between task criteria of completion and its
        execution, so we skip it. The task will be removed anyway
        when the lock is released.

        :raises: TaskLocked
        """
        def _do_delete():
            if not self.zk.exists(Paths.task(task)):
                return
            logger.info('Removing %s', task)
            transaction = self.zk.transaction()
            TaskLayout.delete(transaction, task)
            _check_transaction(transaction.commit())

        if with_lock:
            with self.lock(task) as lock:
                if lock:
                    _do_delete()
                else:
                    raise TaskLocked(task)
        else:
            _do_delete()

    def tasks(self):
        """Provides all enqueued tasks. Ignores invalid tasks."""
        self.zk.ensure_path(Paths.tasks())

        tasks = list(all_tasks(self.zk))

        self.callbacks.before_sorting_by_priority(tasks)

        random.shuffle(tasks)
        ordered_tasks = []
        for task_id in tasks:
            if self.is_locked(task_id):
                continue
            try:
                priority = TaskLayout(self.zk, task_id).priority()
                tag = TaskLayout(self.zk, task_id).tag()
            except (TaskDisappeared, TaskInvalid):
                continue

            if tag != self.tag:
                continue

            heapq.heappush(ordered_tasks, (-priority, task_id))

        while ordered_tasks:
            task = heapq.heappop(ordered_tasks)[1]
            try:
                yield self.load(task)
            except (TaskDisappeared, TaskInvalid):
                continue

    def notify(self, message):
        pass

    def load(self, task_id):
        """Loads task by its id.
        Raises :class:`TaskInvalid` when it is impossible to load a task.
        """
        logger.debug('Loading %s', task_id)
        try:
            self.callbacks.before_task_loading(task_id)
            marshalled = TaskLayout(self.zk, task_id).marshalled()
            task = base.Task.loads(marshalled)
            task._id = task_id
            if not isinstance(task, base.Task):
                raise TaskInvalid('Wrong type of task %s: %s' % (task_id,
                                                                 type(task)))
            return task
        except TaskDisappeared:
            raise
        except kazoo.exceptions.KazooException:
            raise
        except Exception:
            logger.exception('Failed to load task instance')
            _, _, tb = sys.exc_info()
            six.reraise(TaskInvalid, TaskInvalid("Failed to unpickle"), tb)

    @property
    def now(self):
        """Returns current timestamp."""
        return int(time.time())

    @contextlib.contextmanager
    def attempt(self, task):
        """Starts an attempt to perform some action.
        Stores this attempt no matter what result it has produced
        (success or some exception)."""
        attempt = {}

        try:
            attempt['started'] = self.now
            yield
        except:
            attempt['state'] = 'failed'
            raise
        else:
            attempt['state'] = 'success'
        finally:
            attempt['finished'] = self.now
            logger.info('Task %s attempt %s', task, attempt)

    @contextlib.contextmanager
    def lock(self, task):
        """Tries to lock a task. This is a context manager
        that returns the lock when the task is locked succesfully,
        None otherwise.

        :param task: Task that should be locked.
        """
        path = Paths.lock(task)
        self.zk.ensure_path(path)
        lock = kazoo.recipe.lock.Lock(self.zk, path, os.getpid())
        try:
            self.callbacks.before_acquiring_lock(lock)
            lock.acquire(timeout=self.lock_timeout)
        except kazoo.exceptions.NoNodeError:
            logger.debug('Task %s disappeared when locking', task)
            yield None
        except kazoo.exceptions.LockTimeout:
            logger.debug('Task %s is locked', task)
            yield None
        except kazoo.exceptions.KazooException as e:
            logger.exception(e)
            yield None
        else:
            yield lock
        finally:
            lock.release()

    def is_locked(self, task_id):
        """Checks whether task is locked by its id."""
        path = Paths.lock_by_task_id(task_id)
        if not self.zk.exists(path):
            return False
        try:
            lock = self.zk.Lock(path, os.getpid())
            if lock.contenders():
                return True
            else:
                return False
        except kazoo.exceptions.NoNodeError:
            return False

    def execute_and_get_dependencies(self, task):
        """Executes a task. The task should be executed only by one worker
        at the moment. This is done via locking and connection listener that
        stops task execution immediately when connections is lost.

        During task execution it could yield new dependencies. This function
        returns all of these. If task has not yielded any new dependencies
        and is not complete, we believe it is uncompletable so an error
        is raised.

        :raises: TaskFailed
        """
        def _try_delete(task):
            try:
                self.delete(task, with_lock=False)
            except TransactionFailed:
                logger.warning("Did not delete %s", task)

        with self.lock(task) as lock:
            if lock:
                self.assure_valid(task)

                logger.info('Executing %s', task)
                with self.attempt(task):
                    with log_context(task=str(task)):
                        try:
                            for dependency in self.interruptible_fork(task):
                                yield dependency
                        except TaskFailed:
                            logger.exception("Task %s failed", task)
                            self.notify("Task %s failed" % (task))
                            _try_delete(task)
                            raise
                _try_delete(task)
                raise TaskExecuted(task)

    def create_process(self, target, args):
        if self.do_fork:
            return multiprocessing.Process(target=target, args=args)
        else:
            class FakeProcess(object):
                def start(self):
                    target(*args)

                def join(self, timeout=None):
                    pass

                def is_alive(self):
                    return True

                def terminate(self):
                    pass

            return FakeProcess()

    def interruptible_fork(self, task):
        """Runs provided target in an interruptible fashion. Task is
        interrupted immediately if connection to the Zookeeper is lost.
        This prevents task from being executed in parallel."""

        def terminate_process(process, status):
            """Terminates process if connections is lost"""
            if status in (kazoo.client.KazooState.SUSPENDED,
                          kazoo.client.KazooState.LOST):
                logger.warning('Received status %s, terminating', status)
                process.terminate()
                process.join(30)

                # FakeProcess has no pid
                if hasattr(process, 'pid') and process.is_alive():
                    os.kill(process.pid, signal.SIGKILL)

                process.terminated = True

        def run_task_safe(task, queue):
            """Executes target and puts possible exception to the queue."""
            previous_proctitle = getproctitle()
            setproctitle(str(task))
            pickling_support.install()

            def send_error(error, tb):
                if self.do_fork:
                    queue.put(ExecutionErrorInfo(
                        cls=type(error), error=error, traceback=tb
                    ))
                else:
                    six.reraise(type(error), error, tb)

            def send_error_and_raise(error):
                send_error(error, None)
                raise error

            def send(something):
                queue.put(something)

            try:
                logger.info('Executing task %s in a forked process', task)

                with task.run_context() as context:
                    result = task.processed_run(context=context)

                    if not result:
                        logger.debug("Run of %s returned nothing", task)

                    got_new_dependencies = False

                    if isinstance(result, types.GeneratorType):
                        for dependency in result:
                            if task == dependency:
                                send_error_and_raise(TaskCyclic(task))
                            got_new_dependencies = True
                            send(dependency)
                    elif result:
                        logger.warning("Task %s returned non-generator result", task)

                    is_complete = task.complete()
                    # task is incomplete only if is_complete == False
                    # it could be None but that's ok
                    if not got_new_dependencies and is_complete is False:
                        completed = False
                        raise TaskUncompletable()

                    completed = True

                if not completed:
                    raise TaskUncompletable(task)

                send(TaskDone())
            except KeyboardInterrupt:
                send_error(TaskFailed("Keyboard interrupt"), None)
            except TaskUncompletable as e:
                send_error(e, None)
            except Exception:
                _, exc, tb = sys.exc_info()
                send_error(TaskFailed(exc.__class__.__name__ + ':' + str(exc)), tb)
            finally:
                setproctitle(previous_proctitle)

        queue = multiprocessing.Queue()
        process = self.create_process(target=run_task_safe, args=(task, queue))
        process.terminated = False
        process.daemon = True
        terminate_listener = functools.partial(terminate_process, process)
        self.zk.add_listener(terminate_listener)

        self.callbacks.before_executing(process)
        process.start()
        deadline = None if task.timeout is None else (datetime.datetime.now() + task.timeout)
        try:
            while not process.terminated:
                try:
                    self.callbacks.before_receiving_result(queue)
                    result = queue.get(timeout=0.1, block=True)

                    if isinstance(result, TaskDone):
                        break
                    elif isinstance(result, base.Task):
                        yield result
                    elif isinstance(result, ExecutionErrorInfo):
                        six.reraise(result.cls, result.error, result.traceback)
                    else:
                        raise TaskFailed('Task produced %s', result)

                except IOError:
                    raise TaskFailed("Don't know if completed")
                except six.moves.queue.Empty:
                    if not process.is_alive():
                        raise TaskFailed('Process died')
                    elif deadline and datetime.datetime.now() > deadline:
                        process.terminate()
                        raise TaskFailed('Task timed out ({})'.format(task.timeout))
        finally:
            process.join()

        self.zk.remove_listener(terminate_listener)

        if process.terminated:
            raise TaskFailed('Task terminated')

    def is_enqueued(self, task):
        path = Paths.task(task)
        return self.zk.exists(path)

    def enqueue(self, task):
        try:
            enqueue(self.zk, task)
        except TaskAlreadyEnqueued:
            logger.debug('Task already enqueued: %s', task)

    def check_complete(self, task):
        if self.is_locked(Paths.task_id(task)):
            raise TaskLocked(task)

        try:
            return task.complete()
        except Exception as e:
            _, _, traceback = sys.exc_info()
            six.reraise(TaskInvalid, TaskInvalid(e), traceback)

    def assure_proper_version(self, task_):
        if self.version != task_.version:
            raise TaskOutdated(task_)

    def assure_valid(self, task_):
        if not task_.valid:
            raise TaskInvalid(task_)

    def assure_not_cyclic(self, task_):
        if task_.has_cyclic_dependencies():
            raise TaskCyclic(task_)

    def process(self, task):
        """Processes provided task.

        TLDR: it executes the task if there are no
        unsatisfied requirements and it is not already being executed
        by someone else.

        1. The function raises :class:`OutdatedTask` if task has
           wrong version. Worker should ignore these.
        2. The function raises :class:`InvalidTask` if task believes
           it is invalid. There is no need to remove the task as it
           could become valid at some point.
        3. The function raises :class:`CyclicTask` if task has cyclic
           dependencies. This prevents worker from getting into
           an infinite loop.
        4. If any task requirements are already enqueued,
           the function raises :class:`TaskAwaiting`. This prevents
           worker from checking if requirements are
           satisfied again and again.
        5. Unsatisfied requirements are enqueued and :class:`TaskAwaiting`
           is raised.
        6. If the task is not locked and it is complete the function
           tries to lock and delete the task.
        7. Finally the task is executed. All runtime-yielded requirements
           are immediately enqueued.

        :param task: Task to be processed.
        :raises: RecoverableError, kazoo.exceptions.KazooException
        """

        """First we check if we should skip this task at all.
        IMPORTANT NOTE: we should check it again
        once task is locked."""
        self.assure_valid(task)
        self.assure_proper_version(task)
        self.assure_not_cyclic(task)

        for requirement in task.processed_requires():
            if self.is_enqueued(requirement):
                raise TaskAwaiting()
            try:
                if getattr(requirement.__class__, 'complete') == getattr(base.Task, 'complete'):
                    next(requirement.targets())
            except StopIteration:
                raise TaskInvalid('Required task %s has no targets. '
                                  'That will cause infinite loop.'
                                  % (requirement.__class__.__name__))

        if self.check_complete(task):
            logger.info('Task %s is complete', task)
            try:
                self.delete(task, with_lock=True)
            except TransactionFailed:
                logger.warning("Failed to delete task %s", task)
            return

        any_requirement_enqueued = False
        for requirement in task.processed_requires():
            if self.is_enqueued(requirement):
                any_requirement_enqueued = True
                continue

            try:
                complete = self.check_complete(requirement)
            except TaskLocked:
                any_requirement_enqueued = True
                continue
            except Exception as exc:
                _, _, tb = sys.exc_info()
                six.reraise(TaskInvalid, TaskInvalid(exc), tb)

            if not complete:
                self.enqueue(requirement)
                any_requirement_enqueued = True
                continue

        """If any of requirements were already enqueued
        (now or before) we just skip the task as
        it is waiting still."""
        if any_requirement_enqueued:
            raise TaskAwaiting(task)

        else:
            for dependency in self.execute_and_get_dependencies(task):
                self.enqueue(dependency)

    def try_to_process_any_task(self, fail_immediately):
        """Does actions intented for one loop step.
        :raises: NoTasksLeft
        """
        got_anything = False
        for task in self.tasks():
            got_anything = True
            try:
                self.process(task)
            except TaskExecuted:
                logger.debug("Task %s has been executed", task)
                return
            except TaskOutdated:
                try:
                    logger.info("Dropping outdated task %s", task)
                    self.delete(task, with_lock=False)
                except TransactionFailed:
                    logger.warning("Failed to delete task %s", task)

            except (TaskAwaiting, TaskLocked) as e:
                logger.debug(e)
                pass
            except (TaskCyclic, TaskInvalid) as e:
                logger.exception(e)
                if fail_immediately:
                    raise
            except TaskFailed:
                if fail_immediately:
                    raise
            except RecoverableError as e:
                logger.exception(e)
                if fail_immediately:
                    raise
        if not got_anything:
            raise NoTasksLeft()

    def _careless_delete(self, path, recursive):
        try:
            self.zk.delete(path, recursive=recursive)
        except kazoo.exceptions.NoNodeError:
            logger.debug("Failed to remove %s", path)
        except kazoo.exceptions.NotEmptyError:
            logger.debug("%s is not empty", path)

    def cleanup(self):
        for task_id in all_locks(self.zk):
            task_path = Paths.task_by_id(task_id)
            if not self.zk.exists(task_path):
                self._careless_delete(Paths.lock_by_task_id(task_id),
                                      recursive=False)

    def loop(self):
        """Enters main worker loop that processes all enqueued tasks."""
        try:
            while not self.stopped:
                try:
                    self.cleanup()
                    self.try_to_process_any_task(fail_immediately=False)
                except NoTasksLeft:
                    logger.debug("No tasks left")
                except RecoverableError as e:
                    logger.debug("Non-critical error: %s" % (e))
                finally:
                    self.jittered_wait()
        except KeyboardInterrupt:
            raise WorkerStopped()

    def jittered_wait(self):
        jitter = self.polling_jitter * random.random()
        seconds_to_sleep = self.polling_interval + jitter

        logger.debug("Sleeping for %d seconds", seconds_to_sleep)
        time.sleep(seconds_to_sleep)

    def register(self):
        self.zk.create(Paths.worker(self.worker_id),
                       ephemeral=True, makepath=True)

    def run(self):
        logger.info('Starting worker %s in tag %s', self.worker_id, self.tag)
        self.register()
        self.loop()


def _check_transaction(transaction_result):
    for result in transaction_result:
        if isinstance(result, kazoo.exceptions.KazooException):
            raise TransactionFailed(result)


def _enqueue_within_transaction(zk, task):
    """Enqueues task within a transaction."""
    try:
        transaction = zk.transaction()
        TaskLayout.store(transaction, task)
        _check_transaction(transaction.commit())
    except TransactionFailed as exc:
        _, _, tb = sys.exc_info()
        six.reraise(TaskNotEnqueued, TaskNotEnqueued(exc), tb)


def enqueue(zk, task):
    """Enqueues a task.
    Does nothing when task already exists.

    :param zk: Kazoo-compatible zookeeper connection.
    :param task: Task to be enqueued.
    """
    assert isinstance(task, base.Task)
    zk.ensure_path(Paths.tasks())
    path = Paths.task(task)
    if zk.exists(path):
        raise TaskAlreadyEnqueued(task)

    logger.info('Enqueuing task %s', task)
    _enqueue_within_transaction(zk, task)


def execute_async(task, zk):
    """Execute task in a non-blocking manner. Enqueues task
    to the queue but doesn't wait to be done.

    :param task: task to be executed
    :param zk: active connection to Zookeeper
    """
    assert zk.connected
    enqueue(zk, task)


def _run_worker_until_done(zk, do_fork=None):
    """Creates worker and iterates until all tasks are done."""
    worker = Worker(zk, do_fork=do_fork)
    while True:
        try:
            worker.try_to_process_any_task(fail_immediately=True)
        except TaskExecuted:
            logger.info("Task has been recently executed")
            continue
        except NoTasksLeft:
            logger.info("No tasks left")
            break
        except RecoverableError:
            logger.exception('Something happened')
            raise


def execute_sync(task, zk, do_fork=None):
    """Execute task in a blocking manner. Enqueues task
    to the queue and starts a worker that executes all
    tasks until there is none of them.

    There is no guarantee that worker will not
    run other enqueued tasks or the enqueued task won't be
    completed by other worker. To avoid that, provided
    Zookeeper connection should be isolated as well.
    """
    try:
        enqueue(zk, task)
    except TaskAlreadyEnqueued:
        logger.info('Task already exists: %s', task)

    _run_worker_until_done(zk, do_fork=do_fork)


def execute_sync_force(task, zk, do_fork=None):
    execute_sync(task, zk, do_fork)

    worker = Worker(zk, do_fork=do_fork)
    try:
        [_ for _ in worker.execute_and_get_dependencies(task)]
    except TaskExecuted:
        pass


def run_worker(zk, worker_id=None, tag=None):
    """Runs worker.
    Raises exception in the case of unrecoverable error.

    :param zk: active connection to Zookeeper
    """
    assert zk.connected
    worker = Worker(zk, worker_id=worker_id, tag=tag or base.DEFAULT_TAG)
    worker.run()


def purge(zk):
    """Purges the task queue."""
    try:
        zk.delete(Paths.tasks(), recursive=True)
    except kazoo.exceptions.NoNodeError:
        pass


def _get_children_safe(zk, path):
    """Returns list of path's children nodes. Returns empty
    list when path doesn't exist."""
    try:
        return zk.get_children(path)
    except kazoo.exceptions.NoNodeError:
        return []


def all_locks(zk):
    """Yields all locks."""
    for lock in _get_children_safe(zk, Paths.locks()):
        yield lock


def all_tasks(zk):
    """Yields all available tasks."""
    for task in _get_children_safe(zk, Paths.tasks()):
        yield task


def all_workers(zk):
    """Yields all available workers."""
    for worker in _get_children_safe(zk, Paths.workers()):
        yield worker


def status(zk):
    """Returns status dictionary of task queue. The result contains the
    following status keys:

    * 'active' for all active (enqueued) tasks
    * 'failing' for all failing tasks
    * 'complete' for all complete tasks
    * 'non-locked' for all non-locked tasks (not executed or removed at
      the moment)
    * `locked` for all locked tasks
    * 'invalid' for all hashes of invalid tasks
    """
    r = collections.defaultdict(set)
    virtual_worker = Worker(zk)

    statuses = base.Task.Status
    for task_id in all_tasks(zk):
        try:
            task = virtual_worker.load(task_id)
            r[statuses.ENQUEUED].add(task)
            try:
                complete = task.complete()
            except:
                r[base.Task.Status.FAILING].add(task)
            else:
                if complete:
                    r[statuses.COMPLETE].add(task)
                with virtual_worker.lock(task) as lock:
                    if not lock:
                        r[statuses.LOCKED].add(task)
        except TaskInvalid:
            r[statuses.INVALID].add(task_id)
    return r
