from datetime import date, datetime
import collections
import logging

from crypta.lib.python.bt import workflow

logger = logging.getLogger(__name__)


class Exists(workflow.Target):
    """Target that is satisfied once specified table exists."""

    def __init__(self, yt, table):
        self.yt = yt
        self._table = table

    @property
    def path(self):
        return str(self._table)

    def __repr__(self):
        return "table [%s] exists" % (self._table)

    def satisfied(self):
        return self.yt.exists(self._table)


class NotEmpty(workflow.Target):
    """Target that is satisfied once specified table exists and
    is not empty."""

    def __init__(self, yt, table):
        self.yt = yt
        self._table = table

    @property
    def path(self):
        return str(self._table)

    def __repr__(self):
        return "table [%s] is not empty" % (self._table)

    def satisfied(self):
        if self.yt.exists(self._table):
            row_count = self.yt.get_attribute(self._table, 'row_count')
            logger.debug('Table %s exists, has %d rows',
                         self._table, row_count)
            return row_count > 0
        else:
            logger.debug('Table %s doesn\'t exist yet', self._table)
            return False


class HasAttribute(workflow.Target):
    """Target that is satisfied when the specified table exists and
    has specified attribute. Optionally you can check the attribute
    value."""

    def __init__(self, yt, table, attribute, value=None):
        self.yt = yt
        self._table = table
        self._attribute = attribute
        self._value = value

    def __repr__(self):
        return "table [%s] has attribute [%s] = [%s]" % (self._table,
                                                         self._attribute,
                                                         self._value)

    def satisfied(self):
        if not self.yt.exists(self._table):
            logger.debug('No table {} exists'.format(self._table))
            return False
        if not self.yt.has_attribute(self._table, self._attribute):
            logger.debug('No attribute {} exists'.format(self._table))
            return False
        if self._value is None:
            return True
        attr_value = self.yt.get_attribute(self._table, self._attribute)
        if attr_value != self._value:
            logger.debug('Table {} has {} = {}, expected = {}'.format(
                self._table, self._attribute, attr_value, self._value))
            return False
        return True


class NotEmptyAndSorted(NotEmpty):
    """Target that is satisfied when specified table eixsts,
    not empty and sorted."""
    def __repr__(self):
        return "table [%s] is not empty and sorted" % (self._table)

    def satisfied(self):
        non_empty = super(NotEmptyAndSorted, self).satisfied()
        if non_empty and self.yt.is_sorted(self._table):
            return True
        else:
            return False


class ExistsAndSorted(Exists):
    """Target that is satisfied when specified table eixsts
    and is sorted."""
    def __repr__(self):
        return "table [%s] exists and it's sorted" % (self._table)

    def satisfied(self):
        exists = super(ExistsAndSorted, self).satisfied()
        if exists and self.yt.is_sorted(self._table):
            return True
        else:
            return False


class FullyProcessed(workflow.Target):
    """Target that is satisfied once specified table is fully processed.

    To process the table this target provides an API to obtain a chunk
    and commit it back. Currently, this is not tolerant to parallel processing
    and will fail in case of merge conflict.

    The state of processing is stored directly in the processed table via
    attributes.
    """

    Chunk = collections.namedtuple('Chunk', ['lower', 'upper'])

    def __init__(self, yt, table):
        self.yt = yt
        self._table = table
        self._key = 'crypta-processing-state'

    def set_state(self, value):
        self.yt.set_attribute(self._table, self._key, value)

    def get_state(self):
        if not self.yt.has_attribute(self._table, self._key):
            logger.info('Table %s has not been processed before', self._table)
            self.reset()
        else:
            logger.debug('Table %s has been processed before', self._table)

        value = self.yt.get_attribute(self._table, self._key)
        return self.Chunk(lower=value[0], upper=value[1])

    def satisfied(self):
        logger.debug('Checking if table %s is fully processed', self._table)

        if not self.yt.exists(self._table):
            return False

        state = self.get_state()
        logger.debug(state)
        if state.lower == state.upper:
            return True
        else:
            return False

    def get_chunk(self, size):
        state = self.get_state()
        return self.Chunk(lower=state.lower,
                          upper=min(state.lower + size,
                                    state.upper))

    def commit_chunk(self, chunk):
        state = self.get_state()
        assert chunk.lower == state.lower
        self.set_state(self.Chunk(lower=chunk.upper, upper=state.upper))

    def reset(self):
        logger.debug('Resetting state of %s', self._table)
        row_count = self.yt.get_attribute(self._table, 'row_count')
        self.set_state(self.Chunk(lower=0, upper=row_count))


class HasCorrectSources(workflow.Target):
    """Target that is satisfied when table exists and was
    constructed from specified tables.
    """
    def __init__(self, yt, path, sources):
        self.yt = yt
        self.path = path
        self.sources = sources
        self.attribute = 'crypta-table-sources'

    def satisfied(self):
        if self.missing_sources:
            return False
        if self.redundant_sources:
            return False

        return True

    def set_sources(self, sources):
        if not self.yt.exists(self.path):
            raise ValueError()
        self.yt.set_attribute(self.path, self.attribute, sources)

    @property
    def exists_and_has_attribute(self):
        """Checks whether table exists and has the attribute
        to store sources."""
        if not self.yt.exists(self.path):
            return False
        if not self.yt.has_attribute(self.path, self.attribute):
            return False
        return True

    @property
    def missing_sources(self):
        """Returns a list of redundant sources, i.e.
        sources that should have been used but they haven't been."""
        if not self.exists_and_has_attribute:
            return self.sources
        actual_sources = self.yt.get_attribute(self.path, self.attribute)
        return list(set(self.sources) - set(actual_sources))

    @property
    def redundant_sources(self):
        """Returns a list of redundant sources, i.e.
        sources that should have not been used."""
        if not self.exists_and_has_attribute:
            return []
        actual_sources = self.yt.get_attribute(self.path, self.attribute)
        return list(set(actual_sources) - set(self.sources))


class HasNotExpired(workflow.Target):
    """Target that is satisfied when the specified table exists and
    has expiration_datetime/expitation_time attribute. The expiration
    date(time) should be in future. Otherwise table is expired."""

    attribute = '_expiration_time'

    def __init__(self, yt, table, expiration_type='datetime'):
        self.yt = yt
        assert expiration_type in {'datetime', 'date'}
        self._table = table
        if expiration_type == 'datetime':
            self._expiration_time = datetime.now()
        elif expiration_type == 'date':
            self._expiration_time = datetime.combine(date.today(),
                                                     datetime.max.time())

    def satisfied(self):
        if not self.yt.exists(self._table):
            logger.info('No table {} exists'.format(self._table))
            return False
        if not self.yt.has_attribute(self._table, self._attribute):
            logger.info('No attribute {} exists'.format(self._table))
            return False
        attr_value = self.yt.get_attribute(self._table, self.attribute)
        expiration_time = datetime.fromtimestamp(int(attr_value))
        logger.info('Table expiration time: %s (%s)' %
                    (expiration_time.isoformat(), attr_value))
        if self._expiration_time >= expiration_time:
            logger.info('Table expired. Now %s, expiration time %s' %
                        (self._expiration_time.isoformat(),
                         expiration_time.isoformat()))
            return False
        logger.info('Table %s expires in %s (at %s, now %s)' %
                    (self._table,
                     str(expiration_time - self._expiration_time),
                     self._expiration_time.isoformat(),
                     expiration_time.isoformat()))
        return True


class HasSource(workflow.Target):
    """Target that is satisfied when table exists and was
    constructed from specified tables.
    """

    attribute = 'crypta-table-sources'

    def __init__(self, yt, path, sources):
        self.yt = yt
        self.path = path
        self.sources = sources

    def satisfied(self):
        if not self.exists_and_has_attribute:
            logger.info('Not exists')
            return False
        if self.missing_sources:
            logger.info('Not satisfied')
            return False
        logger.info('Satisfied')
        return True

    def set_sources(self):
        if not self.yt.exists(self.path):
            self.yt.create_table(self.path)
        actual_sources = []
        if self.yt.has_attribute(self.path, self.attribute):
            actual_sources = self.yt.get_attribute(self.path, self.attribute)
        values = list(set(actual_sources) | set(self.sources))
        self.yt.set_attribute(self.path, self.attribute,
                              sorted(values))

    @property
    def exists_and_has_attribute(self):
        """Checks whether table exists and has the attribute
        to store sources."""
        if not self.yt.exists(self.path):
            return False
        if not self.yt.has_attribute(self.path, self.attribute):
            return False
        return True

    @property
    def missing_sources(self):
        """Returns a list of redundant sources, i.e.
        sources that should have been used but they haven't been."""
        if not self.exists_and_has_attribute:
            return self.sources
        actual_sources = self.yt.get_attribute(self.path, self.attribute, [])
        return list(set(self.sources) - set(actual_sources))
