import logging
import os
import subprocess
import sys
import time


logger = logging.getLogger(__name__)


class Command:
    def __init__(self, cmd, env=None, attempts=1, timeout_seconds=0, raise_on_failure=True, description=None):
        self.cmd = cmd
        self.env = env
        self.attempts = attempts
        self.timeout_seconds = timeout_seconds
        self.description = description
        self.raise_on_failure = raise_on_failure

    def _run_once(self):
        logger.info("Executing command %s", self.cmd)

        process = subprocess.Popen(self.cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=self.env)
        std_out, std_err = process.communicate()
        exit_code = process.wait()

        if std_out:
            logger.info(std_out)
        if std_err:
            logger.warning(std_err)

        logger.info("Exit code: %s", exit_code)

        return exit_code, std_out, std_err

    def run(self):
        if self.description:
            logger.info(self.description)

        for attempt in range(self.attempts):
            try:
                exit_code, std_out, std_err = self._run_once()

                if exit_code == 0:
                    return
            except Exception:
                logger.exception("Exception")

            logger.error("Failed to execute command %s", self.cmd)

            if attempt < (self.attempts - 1):
                logger.warning("Unsuccessful attempt %s/%s. Sleep for %s s and try again.", attempt + 1, self.attempts, self.timeout_seconds)
                time.sleep(self.timeout_seconds)

        if self.raise_on_failure:
            raise Exception("Failed to execute command {}".format(self.cmd))


def get_cmd_line(binary, config, arg_formatters=None, prefix="--"):
    args = [binary]
    for key, value in config.iteritems():
        if arg_formatters and key in arg_formatters:
            args.extend(arg_formatters[key](key, value))
        else:
            args.extend(("{}{}".format(prefix, key), str(value)))
    return args


def run(cmd, env=None, attempts=1, timeout_seconds=0, raise_on_failure=True, description=None):
    Command(
        cmd,
        env,
        attempts=attempts,
        timeout_seconds=timeout_seconds,
        raise_on_failure=raise_on_failure,
        description=description
    ).run()


def check_call_in_current_env(cmd, **kwargs):
    env = os.environ.copy()
    env.update(kwargs)
    cmd = [unicode(i) for i in cmd]
    subprocess.check_call(cmd, shell=False, stdout=sys.stdout, stderr=sys.stderr, env=env)
