import logging
import posixpath
import re

import requests
import xmltodict

REQUEST_TIMEOUT_SEC = 10

logger = logging.getLogger(__name__)


def get_oauth_headers(token):
    return {"Authorization": "OAuth {}".format(token)} if token else {}


def raise_on_status(response, ok_codes):
    if response.status_code not in ok_codes:
        raise Exception("Bad response HTTP code: {}".format(response.status_code))


def get_api_url(path):
    return "https://c.yandex-team.ru/api/{}".format(path)


def _request(method, url, oauth_token=None, additional_headers=None, data=None, ok_statuses=(requests.codes.ok,)):
    headers = get_oauth_headers(oauth_token)
    headers.update(additional_headers or {})

    logger.debug("Requesting {}".format(url))
    response = requests.request(method, url, headers=headers, data=data, timeout=REQUEST_TIMEOUT_SEC)
    logger.debug("Got response {}".format(response))

    raise_on_status(response, ok_statuses)
    return response


def _get(url, oauth_token=None):
    return _request("GET", url, oauth_token)


def _post(url, data, ok_statuses, oauth_token):
    return _request("POST", url, oauth_token, additional_headers={"Content-Type": "application/json"}, data=data, ok_statuses=ok_statuses)


def _delete(url, oauth_token=None, ok_statuses=(requests.codes.ok,)):
    return _request("DELETE", url, oauth_token, ok_statuses=ok_statuses)


def _get_sorted_lines(url):
    response = _get(url)
    return sorted(response.iter_lines())


def _get_xml(url):
    response = _get(url)
    return xmltodict.parse(response.content)


def _get_sorted_lines_from_path(*args):
    return _get_sorted_lines(get_api_url(posixpath.join(*args)))


def get_host_groups(host):
    return _get_sorted_lines_from_path("hosts2groups", host)


def get_group_hosts(group):
    return _get_sorted_lines_from_path("groups2hosts", group)


def get_host_tags(host):
    return _get_sorted_lines_from_path("hosts2tags", host)


def get_group_tags(group):
    return _get_sorted_lines_from_path("get_group_tags", group)


def get_first_matching_host_group(host, group_regexp=""):
    groups = get_host_groups(host)

    for group in groups:
        if re.match(group_regexp, group):
            return group


def get_group_info(group):
    url = get_api_url("groups/{}".format(group))
    return _get_xml(url)


def get_host_info(host):
    url = get_api_url("hosts/{}?format=json".format(host))
    response = _get(url)
    return response.json()[0]


def get_tag_id(tag, oauth_token):
    url = get_api_url("v1/tags/{}".format(tag))
    response = _get(url, oauth_token)
    return response.json()["id"]


def set_tag(group, tag, oauth_token, ignore_existing=True):
    tag_id = get_tag_id(tag, oauth_token)

    url = get_api_url("v1/groups/{}/tags".format(group))
    data = '{{"tag":{{"id":"{tag_id}"}}}}'.format(tag_id=tag_id)

    success_codes = [requests.codes.created]
    if ignore_existing:
        success_codes.append(requests.codes.unprocessable_entity)

    return _post(url, data, success_codes, oauth_token)


def remove_tag(group, tag, oauth_token):
    url = get_api_url("v1/groups/{}/tags/{}".format(group, tag))
    return _delete(url, oauth_token, (requests.codes.no_content,))
