import logging
import os
import re
import sys
import tempfile

from retry import retry
import tvmauth

from crypta.lib.python import secret_manager
from crypta.lib.python.audience.client import AUDIENCE_INTAPI_TVM_ID
from crypta.lib.python.custom_ml import training_config
from crypta.lib.python.sandbox.client import get_sandbox_client
from crypta.lib.python.swagger import swagger


SECRETS_SINGLETON_CUSTOM_ML = None
CRYPTA_API_URL = {
    'testing': 'https://api-test.crypta.yandex.net/swagger.json',
    'stable': 'https://api.crypta.yandex.net/swagger.json',
}


def get_secrets():
    global SECRETS_SINGLETON_CUSTOM_ML
    if SECRETS_SINGLETON_CUSTOM_ML:
        return SECRETS_SINGLETON_CUSTOM_ML
    SECRETS_SINGLETON_CUSTOM_ML = secret_manager.SecretManager({
        'ROBOT_CRYPTA_SANDBOX_OAUTH': secret_manager.Secret('sec-01csvzg7vtpvb7bgrx18ajsscj', 'token'),
        'PROFILE_TVM_SECRET': secret_manager.Secret('sec-01dq7m7y6xb50x8h5hheh8b4s0', 'client_secret'),
        'CUSTOM_ML_TVM_SECRET': secret_manager.Secret('sec-01fttw9304mryykfqsas90zq26', 'client_secret'),
    })
    return SECRETS_SINGLETON_CUSTOM_ML


@retry(Exception, tries=10, delay=60)
def upload_test_model_to_sandbox(path, attributes, logger, is_folder=False, alias_path=None):
    client = get_sandbox_client(os.getenv('SANDBOX_TOKEN') or get_secrets().get_secret('ROBOT_CRYPTA_SANDBOX_OAUTH'))
    return client.upload_to_sandbox(
        path, 'OTHER_RESOURCE', 'test model', 'CRYPTA', attributes, logger, is_folder=is_folder, alias_path=alias_path,
    )


def get_stderr_logger():
    stderr_logger = logging.getLogger(__name__)
    log_handler = logging.StreamHandler(sys.stderr)
    stderr_logger.handlers = [log_handler]
    stderr_logger.setLevel(logging.DEBUG)
    return stderr_logger


def get_industry_dir_from_industry_name(industry_model_name):
    industry_yt_dir = '_'.join(re.sub(r"([A-Z])", r" \1", industry_model_name).split()).lower()
    return os.path.join(training_config.TRAINABLE_SEGMENTS_DATA_DIRECTORY, industry_yt_dir)


def save_catboost_model(yt_client, new_model, yt_model_path):
    with tempfile.NamedTemporaryFile() as model_file:
        new_model.save_model(model_file.name)
        model_file.seek(0)
        yt_client.write_file(yt_model_path, model_file)


def get_api(env='testing'):
    return swagger(
        CRYPTA_API_URL[env],
        os.getenv('ROBOT_SECRETARY_OAUTH_TOKEN') or get_secrets().get_secret('ROBOT_SECRETARY_OAUTH_TOKEN'),
    )


def get_tvm_client(env=None, tvm_id=training_config.CUSTOM_ML_TVM_ID, tvm_secret_name='CUSTOM_ML_TVM_SECRET'):
    env = env or os.getenv('CRYPTA_ENVIRONMENT')
    tvm_secret = os.getenv(tvm_secret_name) or get_secrets().get_secret(tvm_secret_name)

    tvm_client = tvmauth.TvmClient(
        tvmauth.TvmApiClientSettings(
            self_tvm_id=tvm_id,
            self_secret=tvm_secret,
            dsts={
                'audience': AUDIENCE_INTAPI_TVM_ID,
                'siberia': training_config.SIBERIA_TVM_ID[env],
            },
        )
    )

    return tvm_client
