import contextlib
import ftplib
import logging

from crypta.lib.python.ftp.client import (
    ftp_entry,
    mlsd_parser,
)


logger = logging.getLogger(__name__)


class FtpClient(object):
    def __init__(self, host, port, user, password, root_dir="", secure=False, timeout=None):
        self.host = host
        self.port = port
        self.user = user
        self.password = password
        self.root_dir = root_dir
        self.secure = secure
        self.timeout = timeout

    @classmethod
    def from_proto(cls, proto):
        return cls(
            host=proto.Host,
            port=proto.Port,
            user=proto.User,
            password=proto.Password,
            root_dir=proto.RootDir,
            secure=proto.Secure,
            timeout=proto.TimeoutSeconds,
        )

    @contextlib.contextmanager
    def create_connection(self):
        ftp = ftplib.FTP_TLS(timeout=self.timeout) if self.secure else ftplib.FTP(timeout=self.timeout)
        logger.info("Connect to FTP...")
        ftp.connect(self.host, self.port)
        logger.info("Login to FTP...")
        ftp.login(self.user, self.password)

        if self.secure:
            ftp.prot_p()

        if self.root_dir:
            ftp.cwd(self.root_dir)

        try:
            yield ftp
        finally:
            ftp.quit()

    def mkd(self, directory, ignore_error_perm=False):
        with self.create_connection() as ftp:
            try:
                logger.info("Create directory %s", directory)
                ftp.mkd(directory)
            except ftplib.error_perm as e:
                logger.info(e.message)
                if not ignore_error_perm:
                    raise e

    def nlst(self, *args):
        with self.create_connection() as ftp:
            logger.info("Listing dir %s", " ".join(args))
            response = ftp.nlst(*args)
            logger.info("Found files: %s", response)
            return response

    def mlsd(self, path="."):
        with self.create_connection() as ftp:
            logger.info("Listing dir '%s' in a machine-readable way", path)
            lines = []
            ftp.retrlines("MLSD {path}".format(path=path), lines.append)
            response = [mlsd_parser.parse(line) for line in lines]
            logger.info("Found files: %s", response)
            return response

    @staticmethod
    def _is_dir(ftp, path):
        try:
            old_dir = ftp.pwd()
            ftp.cwd(path)
            ftp.cwd(old_dir)
            return True
        except ftplib.error_perm:
            return False

    def mlsd_fallback(self, path="."):
        with self.create_connection() as ftp:
            logger.info("Listing dir '%s' in a backward compatible way", path)
            response = [
                ftp_entry.FtpEntry(
                    line,
                    {ftp_entry.Attrs.type: ftp_entry.Types.dir if FtpClient._is_dir(ftp, line) else ftp_entry.Types.file}
                ) for line in ftp.nlst(path)
            ]
            logger.info("Found files: %s", response)
            return response

    def list_entries(self, path="."):
        try:
            return self.mlsd(path)
        except ftplib.error_perm:
            return self.mlsd_fallback(path)

    def upload(self, local_file, remote_file):
        with self.create_connection() as ftp:
            logger.info("Upload local file %s as %s to FTP", local_file, remote_file)
            with open(local_file, "rb") as f:
                response = ftp.storbinary("STOR {}".format(remote_file), f)
            logger.info("Finish uploading")
            return response

    def download(self, remote_file, local_file):
        with self.create_connection() as ftp:
            logger.info("Download remote file %s as %s from FTP", remote_file, local_file)
            with open(local_file, "wb") as f:
                response = ftp.retrbinary("RETR {}".format(remote_file), f.write)
            logger.info("Finish downloading")
            return response

    def delete(self, filename):
        with self.create_connection() as ftp:
            logger.info("Remove %s from FTP", filename)
            ftp.delete(filename)
