import os
import re


def reverse_readline(filename, buf_size=8192):
    """a generator that returns the lines of a file in reverse order"""
    with open(filename) as fh:
        segment = None
        offset = 0
        fh.seek(0, os.SEEK_END)
        file_size = fh.tell()
        total_size = remaining_size = fh.tell()
        while remaining_size > 0:
            offset = min(total_size, offset + buf_size)
            fh.seek(file_size - offset)
            buffer = fh.read(min(remaining_size, buf_size))
            remaining_size -= buf_size
            lines = buffer.split('\n')
            # the first line of the buffer is probably not a complete line so
            # we'll save it and append it to the last line of the next buffer
            # we read
            if segment is not None:
                # if the previous chunk starts right from the beginning of line
                # do not concact the segment to the last line of new chunk
                # instead, yield the segment first
                if buffer[-1] != '\n':
                    lines[-1] += segment
                else:
                    yield segment
            segment = lines[0]
            for index in range(len(lines) - 1, 0, -1):
                if len(lines[index]):
                    yield lines[index]
        # Don't yield None if the file was empty
        if segment is not None:
            yield segment


def get_files_sorted_by_ctime(dir_path, filename_regexp):
    return sorted([os.path.join(dir_path, filename) for filename in os.listdir(dir_path) if re.match(filename_regexp, filename)],
                  key=lambda x: os.path.getctime(x),
                  reverse=True)


def read(dir_path, filename_regexp, metric, min_ts):
    values = []

    filenames = get_files_sorted_by_ctime(dir_path, filename_regexp)
    try:
        for filename in filenames:
            for raw_line in reverse_readline(filename):
                line = raw_line.rstrip()
                parts = line.split()

                # graphite logs format - .... <metric> <value> <ts>
                if len(parts) < 3:
                    continue

                if not (parts[-3] == metric or parts[-3].endswith('.{}'.format(metric))):
                    continue

                val, ts = float(parts[-2]), int(parts[-1])

                if ts >= min_ts:
                    values.append(val)
                else:
                    return values

    except Exception as exc:
        raise Exception("Can't parse files {}. Error = {}".format(filenames, exc))

    return values
