import contextlib
import logging
import socket
import time

from crypta.lib.python.graphite.sender import schemas
from crypta.lib.python.swagger import swagger

logger = logging.getLogger(__name__)

DEFAULT_ADDRESS = ("localhost", 42000)
MEGA_GRAPHITE_ADDRESSES = [
    ("mega-graphite-man.search.yandex.net", 2024),
    ("mega-graphite-sas.search.yandex.net", 2024)
]
DEFAULT_TIMEOUT = 5.


def replace_dots(value):
    return value.replace('.', '_') if value else value


class BaseGraphiteSender(object):
    def __init__(self, fqdn=None, schema=schemas.ONE_MIN, root_path=None):
        self.schema = schema
        self.fqdn = fqdn
        self.root_path = root_path

    def _get_metric_parts(self, **kwargs):
        schema = kwargs.get("schema", self.schema)
        fqdn = kwargs.get("fqdn", self.fqdn) or socket.getfqdn()
        root_path = kwargs.get("root_path", self.root_path)

        if schema not in schemas.SCHEMAS:
            raise Exception("Bad graphite schema: {}".format(schema))
        return schema, replace_dots(fqdn), root_path

    @staticmethod
    def _get_ts_from_tuple(item, timestamp):
        return (timestamp if len(item) < 3 else item[2]) or int(time.time())

    def send_metrics(self, metric_tuples, timestamp=None, **kwargs):
        raise NotImplementedError

    def send_metric(self, metric, value, timestamp=None, **kwargs):
        self.send_metrics([(metric, value)], timestamp, **kwargs)


class GraphiteSender(BaseGraphiteSender):
    def __init__(self, graphite_addresses=None, fqdn=None, schema=schemas.ONE_MIN, root_path=None, timeout=DEFAULT_TIMEOUT, dry_run=False):
        super(GraphiteSender, self).__init__(fqdn, schema, root_path)
        self.graphite_addresses = graphite_addresses or [DEFAULT_ADDRESS]
        self.timeout = timeout
        self.dry_run = dry_run

    @staticmethod
    def _get_prefix(schema, fqdn, root_path):
        prefix = "{}.{}".format(schema, fqdn)
        if root_path:
            prefix = "{}.{}".format(prefix, root_path)
        return prefix

    def _get_graphite_string(self, item, timestamp, **kwargs):
        prefix = self._get_prefix(*self._get_metric_parts(**kwargs))
        timestamp = self._get_ts_from_tuple(item, timestamp)
        return "{}.{} {} {}".format(prefix, item[0], item[1], timestamp)

    def _send_to_graphite(self, graphite_str):
        logger.info(graphite_str)

        if self.dry_run:
            return

        for address in self.graphite_addresses:
            with contextlib.closing(socket.create_connection(address, self.timeout)) as sock:
                sock.sendall(graphite_str)

    def send_metrics(self, metric_tuples, timestamp=None, **kwargs):
        self._send_to_graphite("\n".join(self._get_graphite_string(item, timestamp, **kwargs) for item in metric_tuples))


class CryptaAPIGraphiteSender(BaseGraphiteSender):
    def __init__(self, token, fqdn=None, schema=schemas.ONE_MIN, root_path=None, timeout=DEFAULT_TIMEOUT, dry_run=False):
        super(CryptaAPIGraphiteSender, self).__init__(fqdn, schema, root_path)

        self.token = token
        self.timeout = timeout
        self.api = None if dry_run else swagger("https://api.crypta.yandex.net/swagger.json", token)
        self.dry_run = dry_run

    def send_metrics(self, metric_tuples, timestamp=None, **kwargs):
        schema, fqdn, root_path = self._get_metric_parts(**kwargs)
        if not root_path:
            raise Exception("Crypta API does not support empty root path")

        base_params = {
            "frequency": schemas.CRYPTA_API_SCHEMAS[schema],
            "hostname": fqdn,
            "group": root_path
        }
        for item in metric_tuples:
            params = dict(base_params)
            params.update({
                "name": item[0],
                "value": item[1],
                "timestamp": self._get_ts_from_tuple(item, timestamp)
            })
            logger.info(params)

            if not self.dry_run:
                self.api.metric.report(**params).result(timeout=self.timeout)
